import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { FeaturedCollectionComponent } from 'ui/components/collections/featured-collection';
import { requestCollection } from 'actions/collection';

const propTypes = {
    featuredCollection: PropTypes.shape({
        id: PropTypes.string,
        title: PropTypes.string,
        itemsCount: PropTypes.number,
        thumbnails: PropTypes.shape({
            large: PropTypes.string,
            medium: PropTypes.string,
            small: PropTypes.string,
            template: PropTypes.string,
        }).isRequired,
        owner: PropTypes.shape({
            displayName: PropTypes.string,
            id: PropTypes.string,
            name: PropTypes.string,
        }),
    }).isRequired,
    requestCollection: PropTypes.func,
    showFeaturedCollection: PropTypes.bool.isRequired,
    trackEvent: PropTypes.func,
};

const defaultProps = {
    requestCollection() {},
    trackEvent() {},
};

const mapStateToProps = ({ analyticsTracker, collection }) => ({
    trackEvent: analyticsTracker.trackEvent,
    featuredCollection: collection.featuredCollection,
    showFeaturedCollection: !(collection.featuredCollection.id === ''),
});

const mapDispatchToProps = dispatch => ({
    requestCollection(collectionId) {
        dispatch(requestCollection(collectionId));
    },
});

export class FeaturedCollectionContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.handleSelect = this.handleSelect.bind(this);
    }

    componentWillReceiveProps({ featuredCollection, showFeaturedCollection }) {
        if (showFeaturedCollection && !this.props.showFeaturedCollection) {
            this._preloadThumbnail();
            this.props.trackEvent('player_rec_show_collection', {
                // eslint-disable-next-line camelcase
                recommended_collection_id: featuredCollection.id,
            });
        }
    }

    handleSelect(featuredCollection) {
        const {
            trackEvent,
            requestCollection,
        } = this.props;

        requestCollection(featuredCollection.id);
        trackEvent('player_rec_select', {
            // eslint-disable-next-line camelcase
            recommended_vod_view: 'collection',
            // eslint-disable-next-line camelcase
            recommended_collection_id: featuredCollection.id,
        });
    }

    _preloadThumbnail() {
        const { featuredCollection } = this.props;
        const image = new Image();
        image.src = featuredCollection.thumbnails.medium;
    }

    render() {
        const { showFeaturedCollection, featuredCollection } = this.props;

        if (!showFeaturedCollection) {
            return null;
        }

        return (
            <FeaturedCollectionComponent
                thumbnailURL={featuredCollection.thumbnails.medium}
                featuredCollection={featuredCollection}
                onSelect={this.handleSelect}
            />
        );
    }
}

FeaturedCollectionContainer.propTypes = propTypes;
FeaturedCollectionContainer.defaultProps = defaultProps;

export const FeaturedCollection = connect(mapStateToProps, mapDispatchToProps)(FeaturedCollectionContainer);
