import React from 'react';
import PropTypes from 'prop-types';
import assign from 'lodash/assign';
import includes from 'lodash/includes';
import enhanceWithClickOutside from 'react-click-outside';
import { connect } from 'react-redux';
import { ExtensionSettingsButton } from 'ui/components/extensions/extension-settings-button';
import { ExtensionsMenuManager } from 'ui/components/extensions/extensions-menu-manager';
import { showMenu, showExtensionsSubmenu, MENU_STATE_NONE, MENU_STATE_EXTENSIONS } from  'actions/ui';
import { EXTENSION_PROP_TYPE } from 'state/extensions';
import { PLATFORM_MOBILE_WEB } from 'state/env';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { ONLINE_STATUS } from 'state/online-status';
import { TYPE_WATCH_PARTY } from 'state/stream-metadata';
import { LOGGED_IN } from 'state/user';
import {
    EXTENSIONLESS_PLAYER_TYPES,
    EXTENSION_HIDDEN_ANCHOR,
    EXTENSION_MENU_MAIN,
    EXTENSION_MENU_DETAILS,
    EXTENSION_MENU_MANAGE_ACCESS,
    EXTENSION_MENU_REPORT,
} from 'util/extensions';
import {
    requestExtensionIdentityToggle,
} from 'actions/extensions';

export const EXTENSION_SETTINGS_TOOLTIP = 'Extensions';
export const EXTENSION_MENU_ALL = Object.freeze([
    EXTENSION_MENU_MAIN,
    EXTENSION_MENU_DETAILS,
    EXTENSION_MENU_MANAGE_ACCESS,
    EXTENSION_MENU_REPORT,
]);

const propTypes = {
    isLoggedIn: PropTypes.bool.isRequired,
    isSettingsShown: PropTypes.bool.isRequired,
    extensionsSubmenu: PropTypes.string,
    onShowSettings: PropTypes.func,
    onIdentityToggle: PropTypes.func.isRequired,
    playerHeight: PropTypes.number.isRequired,
    loginId: PropTypes.number.isRequired,
    locale: PropTypes.string.isRequired,
    playerType: PropTypes.string.isRequired,
    trackEvent: PropTypes.func.isRequired,
    windowObj: PropTypes.object.isRequired,
    extensions: PropTypes.arrayOf(EXTENSION_PROP_TYPE).isRequired,
    shouldEnableExtensions: PropTypes.bool.isRequired,
    onMenuTransition: PropTypes.func.isRequired,
};

const defaultProps = {
    showSettings() {},
    extensions: [],
};

export const mapStateToProps = state => ({
    extensions: state.extensions.activeOverlayExtensions,
    isLoggedIn: state.user.loggedInStatus === LOGGED_IN,
    isSettingsShown: state.ui.showSettings === MENU_STATE_EXTENSIONS,
    extensionsSubmenu: state.ui.showExtensionsSubmenu,
    shouldEnableExtensions: (
        state.env.platform !== PLATFORM_MOBILE_WEB &&
        !EXTENSIONLESS_PLAYER_TYPES.hasOwnProperty(state.env.playerType) &&
        state.stream.contentType === CONTENT_MODE_LIVE &&
        state.streamMetadata.streamType !== TYPE_WATCH_PARTY &&
        state.playback.contentShowing &&
        state.onlineStatus === ONLINE_STATUS
    ),
    playerHeight: state.playerDimensions.height,
    loginId: state.user.id,
    locale: state.lang.langCode,
    playerType: state.env.playerType,
    trackEvent: state.analyticsTracker.trackEvent,
    windowObj: state.window,
});

const mapDispatchToProps = dispatch => ({
    onShowSettings(show = MENU_STATE_NONE) {
        dispatch(showMenu(show));
    },
    onIdentityToggle(extension) {
        dispatch(requestExtensionIdentityToggle(extension.id));
    },
    onMenuTransition(submenu = EXTENSION_MENU_MAIN) {
        dispatch(showExtensionsSubmenu(submenu));
    },
});

export class ExtensionSettingsContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.handleSettingsButtonClick = this.handleSettingsButtonClick.bind(this);
        this.handleWindowBlur = this.handleWindowBlur.bind(this);
        this.handleIdentityToggleClick = this.handleIdentityToggleClick.bind(this);
        this.trackEvent = this.trackEvent.bind(this);
        this.handleMenuTransition = this.handleMenuTransition.bind(this);
    }

    componentDidMount() {
        const { windowObj } = this.props;
        windowObj.addEventListener('blur', this.handleWindowBlur);
    }

    componentWillUnmount() {
        const { windowObj } = this.props;
        windowObj.removeEventListener('blur', this.handleWindowBlur);
    }

    trackEvent(eventName, extension, props) {
        this.props.trackEvent(eventName, assign(props, {
            /* eslint-disable camelcase */
            extension_id: extension.id,
            extension_version: extension.version,
            extension_anchor: extension.anchor,
            extension_mode: 'viewer',
            locale: this.props.locale,
            login_id: this.props.loginId,
            player_type: this.props.playerType,
            /* eslint-enable */
        }));
    }

    handleWindowBlur() {
        if (this.props.isSettingsShown) {
            this.props.onShowSettings(MENU_STATE_NONE);
        }
    }

    handleClickOutside() {
        if (this.props.isSettingsShown) {
            this.props.onShowSettings(MENU_STATE_NONE);
        }
    }

    handleIdentityToggleClick(extension) {
        this.props.onIdentityToggle(extension);
        this.props.onShowSettings(MENU_STATE_NONE);
    }

    handleSettingsButtonClick() {
        const newMenuState = this.props.isSettingsShown ? MENU_STATE_NONE : MENU_STATE_EXTENSIONS;
        this.props.onShowSettings(newMenuState);
    }

    handleMenuTransition(identifier) {
        if (includes(EXTENSION_MENU_ALL, identifier)) {
            this.props.onMenuTransition(identifier);
        }
    }

    render() {
        const extensions = this.props.extensions.filter(extension => extension.anchor !== EXTENSION_HIDDEN_ANCHOR);
        if (!this.props.shouldEnableExtensions || extensions.length === 0) {
            return null;
        }

        const menuManagerComponent = this.props.isSettingsShown ? (
            <ExtensionsMenuManager
                extensions={extensions}
                submenu={this.props.extensionsSubmenu}
                isLoggedIn={this.props.isLoggedIn}
                onIdentityToggle={this.handleIdentityToggleClick}
                playerHeight={this.props.playerHeight}
                trackEvent={this.trackEvent}
                windowObj={this.props.windowObj}
                handleMenuTransition={this.handleMenuTransition}
            />
        ) : null;

        return (
            <div className="pl-flex">
                <ExtensionSettingsButton
                    onClick={this.handleSettingsButtonClick}
                    tooltip={EXTENSION_SETTINGS_TOOLTIP}
                />
                {menuManagerComponent}
            </div>
        );
    }
}

ExtensionSettingsContainer.propTypes = propTypes;
ExtensionSettingsContainer.defaultProps = defaultProps;

const ExtensionSettingsContainerClickWrapper = enhanceWithClickOutside(ExtensionSettingsContainer);
export const ExtensionSettings = connect(mapStateToProps, mapDispatchToProps)(ExtensionSettingsContainerClickWrapper);
