import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import assign from 'lodash/assign';
import { ExtensionOverlay } from 'ui/components/extensions/extension-overlay';
import { PLATFORM_MOBILE_WEB } from 'state/env';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { LOGGED_IN } from 'state/user';
import { EXTENSION_PROP_TYPE, EXTENSION_MODAL_REQUEST_TYPE } from 'state/extensions';
import { AdContentTypes } from 'actions/ads';
import { showExtensionsSubmenu } from 'actions/ui';
import {
    closeExtensionModal,
    purchaseIntentReceived,
    setExtensionModal,
    updateExtensionIdentityLinking,
} from 'actions/extensions';
import { ONLINE_STATUS } from '../../state/online-status';
import { toggleFullScreen } from 'actions/screen-mode';
import { TYPE_WATCH_PARTY } from 'state/stream-metadata';
import classNames from 'classnames';
import {
    EXTENSIONLESS_PLAYER_TYPES,
    EXTENSION_HIDDEN_ANCHOR,
    EXTENSION_MENU_MANAGE_ACCESS,
} from 'util/extensions';
import { ExtensionsModalManager } from 'ui/components/extensions/extensions-modal-manager';
import { FunctionAction } from 'extension-coordinator/lib/constants';

const DEFAULT_ASPECT_RATIO = 16 / 9;

const propTypes = {
    onExtensionDoubleClick: PropTypes.func.isRequired,
    onIdentityLinked: PropTypes.func.isRequired,
    onModalRequested: PropTypes.func.isRequired,
    onModalClose: PropTypes.func.isRequired,
    onPurchaseIntent: PropTypes.func.isRequired,
    onIdShareRequested: PropTypes.func.isRequired,
    shouldEnableExtensions: PropTypes.bool.isRequired,
    shouldShowExtensions: PropTypes.bool.isRequired,
    extensions: PropTypes.arrayOf(EXTENSION_PROP_TYPE).isRequired,
    modalRequest: EXTENSION_MODAL_REQUEST_TYPE,
    game: PropTypes.string.isRequired,
    width: PropTypes.number.isRequired,
    height: PropTypes.number.isRequired,
    videoResolution: PropTypes.string.isRequired,

    isLoggedIn: PropTypes.bool.isRequired,
    login: PropTypes.string.isRequired,
    loginId: PropTypes.number.isRequired,
    channel: PropTypes.string.isRequired,
    channelId: PropTypes.number.isRequired,
    deviceId: PropTypes.string.isRequired,
    platform: PropTypes.string.isRequired,
    language: PropTypes.string.isRequired,
    locale: PropTypes.string.isRequired,
    playerType: PropTypes.string.isRequired,

    trackEvent: PropTypes.func.isRequired,
};

const mapStateToProps = state => ({
    extensions: state.extensions.activeOverlayExtensions,
    modalRequest: state.extensions.modalRequest,
    game: state.streamMetadata.game,
    width: state.playerDimensions.width,
    height: state.playerDimensions.height,
    videoResolution: state.stats.videoStats.videoResolution,
    shouldEnableExtensions: (
        state.env.platform !== PLATFORM_MOBILE_WEB &&
        !EXTENSIONLESS_PLAYER_TYPES.hasOwnProperty(state.env.playerType) &&
        state.stream.contentType === CONTENT_MODE_LIVE &&
        state.streamMetadata.streamType !== TYPE_WATCH_PARTY &&
        state.playback.contentShowing &&
        state.onlineStatus === ONLINE_STATUS
    ),
    shouldShowExtensions: (
        !state.ui.isMini &&
        state.ads.currentMetadata.contentType === AdContentTypes.NONE &&
        !state.playback.paused
    ),

    isLoggedIn: state.user.loggedInStatus === LOGGED_IN,
    login: state.user.name,
    loginId: state.user.id,
    channel: state.streamMetadata.channel.name,
    channelId: state.streamMetadata.channel.id,
    deviceId: state.env.deviceId,
    platform: state.env.platform,
    language: state.lang.shortCode,
    locale: state.lang.langCode,
    playerType: state.env.playerType,

    trackEvent: state.analyticsTracker.trackEvent,
});

export const mapDispatchToProps = dispatch => ({
    onExtensionDoubleClick() {
        dispatch(toggleFullScreen());
    },
    onIdentityLinked(id, isLinked) {
        dispatch(updateExtensionIdentityLinking(id, isLinked));
    },
    onModalRequested(modalRequest) {
        dispatch(setExtensionModal(modalRequest));
    },
    onModalClose() {
        dispatch(closeExtensionModal());
    },
    onPurchaseIntent(payload, makePurchase) {
        const purchaseIntent = {
            payload,
            makePurchase,
        };
        dispatch(purchaseIntentReceived(purchaseIntent));
    },
    onIdShareRequested() {
        dispatch(showExtensionsSubmenu(EXTENSION_MENU_MANAGE_ACCESS));
    },
});

export class ExtensionsContainerComponent extends React.Component {
    constructor() {
        super(...arguments);

        this._boundTrackEvent = this._trackEvent.bind(this);
        this._boundOnModalRequested = this._onModalRequested.bind(this);
    }

    render() {
        const {
            shouldEnableExtensions,
            shouldShowExtensions,
            extensions,
            game,
            videoResolution,
        } = this.props;
        const aspectRatio = this._getAspectRatio(videoResolution);
        const { width, height } = this._computeDimensions(this.props.width, this.props.height, aspectRatio);
        const trackingProperties = this._getTrackingProperties();

        if (!shouldEnableExtensions || !extensions.length) {
            return null;
        }

        // Current UX is for a single overlay in the player. Expand this when multiple are supported.
        const extension = extensions[0];
        const EXTENSION_CONTAINER_CLASSES = classNames({
            'extension-container': true,
            // Removes from DOM
            hide: extension.anchor === EXTENSION_HIDDEN_ANCHOR,
            // Does not remove from DOM
            invisible: !shouldShowExtensions,
        });

        const extensionModal = (
            <ExtensionsModalManager
                playerHeight={height}
                extensions={extensions}
                modalRequest={this.props.modalRequest}
                onModalClose={this.props.onModalClose}
            />
        );

        return (
            <div className={EXTENSION_CONTAINER_CLASSES}>
                <div className="extension-overlays">
                    <ExtensionOverlay
                        ref={this._boundExtensionOverlayRefHandler}
                        width={width}
                        height={height}
                        key={extension.id}
                        extension={extension}
                        game={game}
                        isLoggedIn={this.props.isLoggedIn}
                        language={this.props.language}
                        locale={this.props.locale}
                        loginId={this.props.loginId}
                        onIdentityLinked={this.props.onIdentityLinked}
                        onModalRequested={this._boundOnModalRequested}
                        trackingProperties={trackingProperties}
                        onDoubleClick={this.props.onExtensionDoubleClick}
                        onBeginPurchase={this.props.onPurchaseIntent}
                    />
                </div>
                {extensionModal}
            </div>
        );
    }

    _trackEvent(eventName, props) {
        const extension = this.props.extensions[0];

        this.props.trackEvent(eventName, assign(props, {
            /* eslint-disable camelcase */
            extension_id: extension.id,
            extension_version: extension.version,
            extension_anchor: extension.anchor,
            extension_mode: 'viewer',
            locale: this.props.locale,
            login_id: this.props.loginId,
            player_type: this.props.playerType,
            /* eslint-enable */
        }));
    }

    _getTrackingProperties() {
        return {
            login: this.props.login,
            channel: this.props.channel,
            channelId: this.props.channelId,
            deviceId: this.props.deviceId,
            platform: this.props.platform,
            playerType: this.props.playerType,
        };
    }

    _computeDimensions(width, height, aspectRatio) {
        const tooWide = (width / height > aspectRatio);

        return {
            width: (tooWide ? Math.ceil(height * aspectRatio) : width),
            height: (tooWide ? height : Math.ceil(width / aspectRatio)),
        };
    }

    _getAspectRatio(resolutionString) {
        const resolutionRegex = /(\d+)\D*(\d+)/g;
        const match = resolutionRegex.exec(resolutionString);
        if (!match) {
            return DEFAULT_ASPECT_RATIO;
        }

        const videoWidth = parseInt(match[1], 10);
        const videoHeight = parseInt(match[2], 10);

        if (!videoHeight) {
            return DEFAULT_ASPECT_RATIO;
        }

        return videoWidth / videoHeight;
    }

    _onModalRequested(modalRequest) {
        if (modalRequest.action === FunctionAction.IdShareRequest) {
            this.props.onIdShareRequested();
            return;
        }

        this.props.onModalRequested(modalRequest);
    }
}

ExtensionsContainerComponent.propTypes = propTypes;

export const ExtensionsContainer = connect(mapStateToProps, mapDispatchToProps)(ExtensionsContainerComponent);
