import { connect } from 'react-redux';
import React from 'react';
import PropTypes from 'prop-types';
import { enableNotifications, disableNotifications, promptLoginModal,
         NOTIFICATIONS_ON, NOTIFICATIONS_OFF, FOLLOW_AND_NOTIFICATIONS_ON,
         NOTIFICATIONS_BUTTON_CLICK, NOTIFICATIONS_BUTTON_VIEW } from 'actions/follow';
import { LOGGED_IN } from 'state/user';
import assign from 'lodash/assign';
import { TWITCH_SIGNUP_URL } from 'settings';
import { PLAYER_SITE } from 'util/player-type';
import { TYPE_DEAD_LTV } from 'state/stream-metadata';
import { CONTENT_SCREEN, VOD_RECOMMENDATION_SCREEN } from 'actions/screen';

function getDisplayName(Component) {
    return Component.displayName || Component.name || 'Component';
}

const propTypes = {
    analytics: PropTypes.object,
    channel: PropTypes.shape({
        id: PropTypes.number,
        name: PropTypes.string,
    }).isRequired,
    disableNotifications: PropTypes.func,
    enableNotifications: PropTypes.func,
    playerType: PropTypes.string.isRequired,
    promptLoginModal: PropTypes.func,
    showFollowNotification: PropTypes.bool.isRequired,
    relationship: PropTypes.shape({
        following: PropTypes.bool,
        notificationsEnabled: PropTypes.bool,
    }).isRequired,
    user: PropTypes.shape({
        id: PropTypes.number,
        loggedIn: PropTypes.bool,
    }).isRequired,
    win: PropTypes.object.isRequired,
    withInfo: PropTypes.bool.isRequired,
};

const defaultProps = {
    analytics: {
        trackEvent() {},
    },
    enableNotifications() {},
    disableNotifications() {},
    promptLoginModal() {},
};

export const mapStateToProps = ({
    analyticsTracker,
    env,
    follow,
    playback,
    playerOptions,
    streamMetadata,
    user,
    screen,
    window: win,
    ui,
}) => ({
    analytics: analyticsTracker,
    channel: {
        id: streamMetadata.channel.id,
        avatarUrl: streamMetadata.channel.logo,
        name: streamMetadata.channel.displayName,
    },
    playerType: env.playerType,
    showFollowNotification: (
        follow.showFollowNotification &&
        !ui.isMini &&
        (playback.ended || streamMetadata.streamType === TYPE_DEAD_LTV) &&
        (screen[0] === VOD_RECOMMENDATION_SCREEN || screen[0] === CONTENT_SCREEN)
    ),
    relationship: {
        notificationsEnabled: follow.notificationsEnabled,
        following: follow.following,
    },
    win,
    withInfo: playerOptions.showInfo,
    user: {
        id: user.id,
        loggedIn: user.loggedInStatus === LOGGED_IN,
    },
});

export const mapDispatchToProps = dispatch => ({
    enableNotifications(userId, channelId) {
        dispatch(enableNotifications(userId, channelId));
    },
    disableNotifications(userId, channelId) {
        dispatch(disableNotifications(userId, channelId));
    },
    promptLoginModal(channelName) {
        dispatch(promptLoginModal(channelName));
    },
});

export function makeFollowPanel(FollowPanel) {
    class FollowPanelWrapper extends React.Component {
        constructor() {
            super(...arguments);
            this.enableNotifications = this.enableNotifications.bind(this);
            this.disableNotifications = this.disableNotifications.bind(this);
            this.promptUserLogin = this.promptUserLogin.bind(this);
        }

        componentWillReceiveProps(nextProps) {
            if (nextProps.showFollowNotification && !this.props.showFollowNotification) {
                this._logInteraction(NOTIFICATIONS_BUTTON_VIEW);
            }
        }

        _invokeIfLoggedIn(method) {
            const { user, channel } = this.props;

            if (user.loggedIn) {
                method(user.id, channel.id);
            } else {
                this.promptUserLogin();
            }
        }

        _logInteraction(eventName, event = {}) {
            const { analytics, relationship, playerType } = this.props;

            analytics.trackEvent(eventName, assign(event, {
                /* eslint-disable camelcase */
                follow_state: relationship.following,
                notif_state: relationship.notificationsEnabled,
                /* eslint-enable */
                src: playerType,
            }));
        }

        enableNotifications() {
            const { enableNotifications, relationship } = this.props;
            const action = relationship.following ? NOTIFICATIONS_ON : FOLLOW_AND_NOTIFICATIONS_ON;
            this._invokeIfLoggedIn(enableNotifications);
            this._logInteraction(NOTIFICATIONS_BUTTON_CLICK, { action });
        }

        disableNotifications() {
            const { disableNotifications } = this.props;
            this._invokeIfLoggedIn(disableNotifications);
            this._logInteraction(NOTIFICATIONS_BUTTON_CLICK, {
                action: NOTIFICATIONS_OFF,
            });
        }

        promptUserLogin() {
            const { win, playerType, promptLoginModal, channel } = this.props;

            if (playerType === PLAYER_SITE) {
                promptLoginModal(channel.name);
            } else {
                win.open(TWITCH_SIGNUP_URL, '_blank');
            }
        }

        render() {
            const {
                channel,
                showFollowNotification,
                relationship,
                withInfo,
            } = this.props;

            if (!showFollowNotification) {
                return null;
            }

            return (
                <FollowPanel
                    channelName={channel.name}
                    disable={this.disableNotifications}
                    enable={this.enableNotifications}
                    notificationsEnabled={relationship.notificationsEnabled}
                    withInfo={withInfo}
                />
            );
        }
    }

    FollowPanelWrapper.propTypes = propTypes;
    FollowPanelWrapper.defaultProps = defaultProps;
    FollowPanelWrapper.displayName = `WrappedFollowPanel(${getDisplayName(FollowPanel)})`;

    return connect(mapStateToProps, mapDispatchToProps)(FollowPanelWrapper);
}
