import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import includes from 'lodash/includes';
import { AgeGateOverlay as AgeGateOverlayComponent } from '../../components/overlays/age-gate-overlay';
import { localStore } from 'util/storage';
import { play, pause, mutePlayer } from 'actions/video-api';
import { contentIsShowing } from 'actions/playback';
import { requestAds, DEFAULT_AD_DURATION } from 'actions/ads';
import { PREROLL } from 'ads/ima-tags';

export const KEY_MATURE = 'mature';
export const KEY_AGE_GATE = 'age_gate';
export const AGE_GATE_CHANNEL_AGES =  Object.freeze({
    budlight: 21,
    budlightcanada: 19,
});

const propTypes = {
    playback: PropTypes.object.isRequired,
    streamMetadata: PropTypes.object.isRequired,
    mutePlayer: PropTypes.func.isRequired,
    contentIsShowing: PropTypes.func.isRequired,
    play: PropTypes.func.isRequired,
    pause: PropTypes.func.isRequired,
    requestAds: PropTypes.func.isRequired,
};

const defaultProps = {
    playback() {},
    streamMetadata() {},
    mutePlayer() {},
    contentIsShowing() {},
    play() {},
    pause() {},
    requestAds() {},
};

const mapStateToProps = ({ playback, streamMetadata }) => ({
    playback,
    streamMetadata,
});

const mapDispatchToProps = dispatch => ({
    mutePlayer(muted, automated) {
        dispatch(mutePlayer(muted, automated));
    },
    contentIsShowing() {
        dispatch(contentIsShowing());
    },
    play() {
        dispatch(play());
    },
    pause() {
        dispatch(pause());
    },
    requestAds(rollType, duration) {
        dispatch(requestAds(rollType, duration));
    },
});

export class AgeGateOverlayContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            showAgeGateOverlay: false,
        };

        this.handleAgeGatePassed = this.handleAgeGatePassed.bind(this);
    }

    componentWillReceiveProps(nextProps) {
        const shouldShow = this._shouldShowOverlay(nextProps);
        const overlayIsShowing = this.state.showAgeGateOverlay;
        if (shouldShow && !overlayIsShowing) {
            this.setState({ showAgeGateOverlay: true });
            this.props.mutePlayer(true, true);
            this.props.pause();
            this.props.contentIsShowing();
        } else if (!shouldShow && overlayIsShowing) {
            this.setState({ showAgeGateOverlay: false });
        }
    }

    _shouldShowOverlay({ streamMetadata }) {
        const ageGatesPassed = localStore.get(KEY_AGE_GATE, {});

        return (
            includes(Object.keys(AGE_GATE_CHANNEL_AGES), streamMetadata.channel.name) &&
            !ageGatesPassed[streamMetadata.channel.name]
        );
    }

    render() {
        const { streamMetadata } = this.props;

        if (this.state.showAgeGateOverlay) {
            return (
                <AgeGateOverlayComponent
                    handleAgeGatePassed={this.handleAgeGatePassed}
                    channelName={streamMetadata.channel.name}
                    requiredAge={AGE_GATE_CHANNEL_AGES[streamMetadata.channel.name]}
                />
            );
        }

        return null;
    }

    handleAgeGatePassed() {
        const { playback, streamMetadata } = this.props;
        const ageGates = localStore.get(KEY_AGE_GATE, {});
        ageGates[streamMetadata.channel.name] = true;
        localStore.set(KEY_AGE_GATE, ageGates);

        this.setState({
            showAgeGateOverlay: false,
        });

        this.props.play();
        this.props.mutePlayer(false, true);

        if (playback.autoplay) {
            this.props.requestAds(PREROLL, DEFAULT_AD_DURATION);
        }
    }
}

AgeGateOverlayContainer.propTypes = propTypes;
AgeGateOverlayContainer.defaultProps = defaultProps;
export const AgeGateOverlay = connect(mapStateToProps, mapDispatchToProps)(AgeGateOverlayContainer);
