import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { LoadingOverlay as LoadingOverlayComponent } from 'ui/components/overlays/loading-overlay';
import { ADVERTISEMENT_SCREEN } from 'actions/screen';
import { CONNECTED, CONNECTING } from 'state/chromecast';
import { play } from 'actions/video-api';

const propTypes = {
    isCasting: PropTypes.bool.isRequired,
    isPaused: PropTypes.bool.isRequired,
    isLoading: PropTypes.bool.isRequired,
    isAdvertisementScreen: PropTypes.bool.isRequired,
    playStream: PropTypes.func,
    windowObj: PropTypes.object.isRequired,
};

const defaultProps = {
    playStream() {},
};

export const mapStateToProps = ({ chromecast, playback, screen, window: windowObj }) => ({
    isCasting: (
        chromecast.castingState === CONNECTED ||
        chromecast.castingState === CONNECTING
    ),
    isPaused: playback.paused,
    isLoading: playback.isLoading,
    isAdvertisementScreen: screen[0] === ADVERTISEMENT_SCREEN,
    windowObj,
});

export const mapDispatchToProps = dispatch => ({
    playStream() {
        dispatch(play());
    },
});

export class LoadingOverlayContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            isUserOffline: false,
            wasPausedOnOffline: false,
        };

        this._onOfflineEvent = this._onOfflineEvent.bind(this);
        this._onOnlineEvent = this._onOnlineEvent.bind(this);
    }

    componentDidMount() {
        this.props.windowObj.addEventListener('offline', this._onOfflineEvent);
        this.props.windowObj.addEventListener('online', this._onOnlineEvent);
    }

    componentWillUnmount() {
        this.props.windowObj.removeEventListener('offline', this._onOfflineEvent);
        this.props.windowObj.removeEventListener('online', this._onOnlineEvent);
    }

    _onOfflineEvent() {
        this.setState({
            isUserOffline: true,
            wasPausedOnOffline: this.props.isPaused,
        });
    }

    _onOnlineEvent() {
        if (!this.state.wasPausedOnOffline) {
            this.props.playStream();
        }

        this.setState({
            isUserOffline: false,
            wasPausedOnOffline: false,
        });
    }

    render() {
        const {
            isLoading,
            isPaused,
            isCasting,
            isAdvertisementScreen,
        } = this.props;

        const showLoadingIndicator = (
            isLoading &&
            !isPaused &&
            !isCasting &&
            !isAdvertisementScreen
        );

        if (!showLoadingIndicator) {
            return null;
        }

        return (
            <LoadingOverlayComponent
                shouldShowOfflineOverlay={this.state.isUserOffline}
            />
        );
    }
}

LoadingOverlayContainer.propTypes = propTypes;
LoadingOverlayContainer.defaultProps = defaultProps;
export const LoadingOverlay = connect(mapStateToProps, mapDispatchToProps)(LoadingOverlayContainer);
