import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { MatureOverlay as MatureOverlayComponent } from '../../components/overlays/mature-overlay';
import { localStore } from 'util/storage';
import includes from 'lodash/includes';
import { play, pause, mutePlayer } from 'actions/video-api';
import { contentIsShowing } from 'actions/playback';
import { requestAds, DEFAULT_AD_DURATION } from 'actions/ads';
import { PLAYER_ANIMATED_THUMBNAILS } from 'util/player-type';
import { PREROLL } from 'ads/ima-tags';
import { AGE_GATE_CHANNEL_AGES } from './age-gate-overlay';

export const KEY_MATURE = 'mature';

const propTypes = {
    playback: PropTypes.object,
    playerType: PropTypes.string,
    streamMetadata: PropTypes.object,
    mutePlayer: PropTypes.func,
    contentIsShowing: PropTypes.func,
    play: PropTypes.func,
    pause: PropTypes.func,
    requestAds: PropTypes.func,
};

const mapStateToProps = ({ env, playback, streamMetadata }) => ({
    playback,
    playerType: env.playerType,
    streamMetadata,
});

const mapDispatchToProps = dispatch => ({
    mutePlayer(muted, automated) {
        dispatch(mutePlayer(muted, automated));
    },
    contentIsShowing() {
        dispatch(contentIsShowing());
    },
    play() {
        dispatch(play());
    },
    pause() {
        dispatch(pause());
    },
    requestAds(rollType, duration) {
        dispatch(requestAds(rollType, duration));
    },
});

export class MatureOverlayContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            showMatureOverlay: false,
        };

        this.handleMatureAccepted = this.handleMatureAccepted.bind(this);
    }

    componentWillReceiveProps(nextProps) {
        const shouldShow = this._shouldShowOverlay(nextProps);
        const overlayIsShowing = this.state.showMatureOverlay;

        if (shouldShow && !overlayIsShowing) {
            this.setState({ showMatureOverlay: true });
            this.props.mutePlayer(true, true);
            this.props.pause();
            this.props.contentIsShowing();
        } else if (!shouldShow && overlayIsShowing) {
            this.setState({ showMatureOverlay: false });
        }
    }

    _shouldShowOverlay({ streamMetadata }) {
        const isMature = streamMetadata.channel.mature;

        return (
            isMature &&
            !localStore.get(KEY_MATURE) &&
            this.props.playerType !== PLAYER_ANIMATED_THUMBNAILS &&
            !includes(Object.keys(AGE_GATE_CHANNEL_AGES), streamMetadata.channel.name)
        );
    }

    render() {
        if (this.state.showMatureOverlay) {
            return (
                <MatureOverlayComponent
                    handleMatureAccepted={this.handleMatureAccepted}
                />
            );
        }
        return null;
    }

    handleMatureAccepted() {
        const { playback } = this.props;
        localStore.set(KEY_MATURE, true);

        this.setState({
            showMatureOverlay: false,
        });

        this.props.play();
        this.props.mutePlayer(false, true);

        if (playback.autoplay) {
            this.props.requestAds(PREROLL, DEFAULT_AD_DURATION);
        }
    }
}

MatureOverlayContainer.propTypes = propTypes;
export const MatureOverlay = connect(mapStateToProps, mapDispatchToProps)(MatureOverlayContainer);
