import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { twitchHost } from 'settings';
import { contentIsShowing, clearQualityRestrictedError } from 'actions/playback';
import { pushScreen, popScreen, QUALITY_RESTRICTED_SCREEN } from 'actions/screen';
import { QualityRestrictedOverlay as QualityRestricted } from 'ui/components/overlays/quality-restricted-overlay';

const propTypes = {
    channelName: PropTypes.string.isRequired,
    contentIsShowing: PropTypes.func.isRequired,
    popScreen: PropTypes.func.isRequired,
    shouldRenderQualityOverlay: PropTypes.bool.isRequired,
    dispatchQualityRestrictedScreen: PropTypes.func.isRequired,
    clearQualityRestrictedError: PropTypes.func.isRequired,
};

const defaultProps = {
    shouldRenderQualityOverlay: false,
};

export const mapStateToProps = ({ playback, streamMetadata }) => ({
    shouldRenderQualityOverlay: playback.restrictedQualityError,
    channelName: streamMetadata.channel.name,
});

export const mapDispatchToProps = dispatch => ({
    contentIsShowing() {
        dispatch(contentIsShowing());
    },
    dispatchQualityRestrictedScreen() {
        dispatch(pushScreen(QUALITY_RESTRICTED_SCREEN));
    },
    popScreen() {
        dispatch(popScreen());
    },
    clearQualityRestrictedError() {
        dispatch(clearQualityRestrictedError());
    },
});

export class QualityRestrictedContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.onClose = this.onClose.bind(this);
    }

    componentDidMount() {
        this.props.contentIsShowing();
    }

    componentWillReceiveProps(nextProps) {
        if (nextProps.shouldRenderQualityOverlay) {
            this.props.dispatchQualityRestrictedScreen();
        }
    }

    render() {
        const { channelName, shouldRenderQualityOverlay } = this.props;
        const subscribeUrl = `${twitchHost}/${channelName}/subscribe?ref=chansub_overlay_subscribe`;

        if (!shouldRenderQualityOverlay) {
            return null;
        }

        return (
            <QualityRestricted
                channelName={channelName}
                subscribeUrl={subscribeUrl}
                onClose={this.onClose}
            />
        );
    }

    onClose() {
        const { popScreen, clearQualityRestrictedError } = this.props;
        clearQualityRestrictedError();
        popScreen();
    }
}

QualityRestrictedContainer.propTypes = propTypes;
QualityRestrictedContainer.defaultProps = defaultProps;
export const QualityRestrictedOverlay = connect(mapStateToProps, mapDispatchToProps)(QualityRestrictedContainer);
