import React from 'react';
import PropTypes from 'prop-types';
import TransitionGroup from 'react-transition-group/TransitionGroup';
import { connect } from 'react-redux';
import { mutePlayer } from 'actions/video-api';
import { trackEvent } from 'actions/analytics-tracker';
import { AnimatedUnmuteOverlay } from 'ui/components/overlays/unmute-overlay';
import includes from 'lodash/includes';
import { pushScreen, popScreen, UNMUTE_SCREEN } from 'actions/screen';

const OVERLAY_DURATION = 5000; // ms
const PLAYER_CLICK_TRACKING_EVENT = 'player_click';
const CLICK_TRACKING_COMPONENT = 'unmute_overlay';

const propTypes = {
    unmutePlayer: PropTypes.func.isRequired,
    autoplayEnabled: PropTypes.bool.isRequired,
    isMobileWeb: PropTypes.bool.isRequired,
    isPlaying: PropTypes.bool.isRequired,
    hasPlayed: PropTypes.bool.isRequired,
    isMuted: PropTypes.bool.isRequired,
    setTimeout: PropTypes.func.isRequired,
    clearTimeout: PropTypes.func.isRequired,
    trackOverlayClick: PropTypes.func.isRequired,
    pushUnmuteScreen: PropTypes.func.isRequired,
    popScreen: PropTypes.func.isRequired,
    screenStack: PropTypes.arrayOf(PropTypes.string).isRequired,
};

export const mapStateToProps = ({ playback, env, window, screen }) => ({
    autoplayEnabled: playback.autoplay,
    isMobileWeb: env.isMobileLocation,
    isPlaying: playback.playing,
    hasPlayed: playback.hasPlayed,
    isMuted: playback.muted,
    setTimeout: window.setTimeout,
    clearTimeout: window.clearTimeout,
    screenStack: screen,
});

export const mapDispatchToProps = dispatch => ({
    unmutePlayer() {
        dispatch(mutePlayer(false));
    },
    trackOverlayClick() {
        dispatch(trackEvent(PLAYER_CLICK_TRACKING_EVENT, {
            component: CLICK_TRACKING_COMPONENT,
        }));
    },
    pushUnmuteScreen() {
        dispatch(pushScreen(UNMUTE_SCREEN));
    },
    popScreen() {
        dispatch(popScreen());
    },
});

function isFirstPlay(prevProps, nextProps) {
    return (
        !prevProps.hasPlayed &&
        !prevProps.isPlaying &&
        nextProps.isPlaying
    );
}

export class UnmuteOverlayContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.state = { hideTimeoutId: 0 };
        this.handleClick = this.handleClick.bind(this);
        this.startTimeout = this.startTimeout.bind(this);
        this.cancelTimeout = this.cancelTimeout.bind(this);
        this.timeoutHandler = this.timeoutHandler.bind(this);
    }

    componentDidMount() {
        if (
            this.props.isMuted &&
            this.props.autoplayEnabled &&
            this.props.isPlaying &&
            this.props.isMobileWeb &&
            !includes(this.props.screenStack, UNMUTE_SCREEN)
        ) {
            this.props.pushUnmuteScreen();
        }
    }

    // eslint-disable-next-line complexity
    componentWillReceiveProps(nextProps) {
        if (!nextProps.autoplayEnabled || !nextProps.isMobileWeb) {
            return;
        }

        if (
            (!nextProps.isMuted || !nextProps.isPlaying) &&
            nextProps.screenStack[0] === UNMUTE_SCREEN
        ) {
            this.props.popScreen();
        }

        if (
            nextProps.isMuted &&
            isFirstPlay(this.props, nextProps) &&
            !includes(nextProps.screenStack, UNMUTE_SCREEN)
        ) {
            this.props.pushUnmuteScreen();
        }

        if (
            this.props.screenStack[0] === UNMUTE_SCREEN &&
            nextProps.screenStack[0] !== UNMUTE_SCREEN
        ) {
            this.cancelTimeout();
        }

        if (
            this.props.screenStack[0] !== UNMUTE_SCREEN &&
            nextProps.screenStack[0] === UNMUTE_SCREEN
        ) {
            this.startTimeout();
        }
    }

    render() {
        const animatedUnmuteOverlay = (
            <AnimatedUnmuteOverlay
                onClick={this.handleClick}
            />
        );

        const overlay = this.props.screenStack[0] === UNMUTE_SCREEN ? animatedUnmuteOverlay : null;

        return (
            <TransitionGroup component={'div'}>
                {overlay}
            </TransitionGroup>
        );
    }

    timeoutHandler() {
        const { screenStack, popScreen } = this.props;

        if (screenStack[0] === UNMUTE_SCREEN) {
            popScreen();
        }
    }

    startTimeout() {
        const { setTimeout, clearTimeout } = this.props;

        clearTimeout(this.state.hideTimeoutId);
        const hideTimeoutId = setTimeout(this.timeoutHandler, OVERLAY_DURATION);

        this.setState({ hideTimeoutId });
    }

    cancelTimeout() {
        const { clearTimeout } = this.props;

        clearTimeout(this.state.hideTimeoutId);
        this.setState({ hideTimeoutId: 0 });
    }

    handleClick() {
        const { unmutePlayer, trackOverlayClick } = this.props;

        unmutePlayer();
        trackOverlayClick();
    }
}

UnmuteOverlayContainer.propTypes = propTypes;

export const UnmuteOverlay = connect(mapStateToProps, mapDispatchToProps)(UnmuteOverlayContainer);
