import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { VOD_RECOMMENDATION_SCREEN, popScreen } from 'actions/screen';
import { OfflineRecommendations } from 'ui/components/recommendations/offline-recommendations';
import { clearRecommendedVODs, OFFLINE_RECOMMENDATIONS_TYPE } from 'actions/recommendations';
import { selectRecommendedVideo } from 'actions/playback';

const propTypes = {
    recommendedVideos: PropTypes.array.isRequired,
    recommendationType: PropTypes.string.isRequired,
    windowObj: PropTypes.object.isRequired,
    showRecommendations: PropTypes.bool.isRequired,
    selectRecommendedVideo: PropTypes.func.isRequired,
    popScreen: PropTypes.func.isRequired,
    clearRecommendedVODs: PropTypes.func.isRequired,
    i18n: PropTypes.object.isRequired,
    playerDimensions: PropTypes.shape({
        height: PropTypes.number,
        width: PropTypes.number,
    }).isRequired,
    trackEvent: PropTypes.func.isRequired,
    streamHasPlayed: PropTypes.bool.isRequired,
};

export const mapStateToProps = ({
    lang,
    screen,
    recommendations,
    window: windowObj,
    playerDimensions,
    analyticsTracker,
    playback,
}) => ({
    recommendedVideos: recommendations.videos,
    recommendationType: recommendations.type,
    windowObj,
    showRecommendations: (
        screen[0] === VOD_RECOMMENDATION_SCREEN &&
        playback.ended
    ),
    i18n: lang,
    playerDimensions,
    trackEvent: analyticsTracker.trackEvent,
    streamHasPlayed: playback.hasPlayed,
});

export const mapDispatchToProps = dispatch => ({
    selectRecommendedVideo(vodID) {
        dispatch(selectRecommendedVideo(vodID));
    },

    popScreen() {
        dispatch(popScreen());
    },

    clearRecommendedVODs() {
        dispatch(clearRecommendedVODs());
    },
});

export class OfflineRecommendationsContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.handleSelectVOD = this.handleSelectVOD.bind(this);
        this.handleExitRecommendations = this.handleExitRecommendations.bind(this);
    }

    render() {
        const {
            recommendedVideos,
            windowObj,
            showRecommendations,
            i18n,
            playerDimensions,
            recommendationType,
            trackEvent,
            streamHasPlayed,
        } = this.props;

        if (recommendedVideos.length > 0) {
            this._preloadVideoThumbnails();
        }

        if (
            !showRecommendations ||
            recommendedVideos.length === 0
        ) {
            return null;
        }

        switch (recommendationType) {
        case OFFLINE_RECOMMENDATIONS_TYPE:
            return (
                <OfflineRecommendations
                    videos={recommendedVideos}
                    i18n={i18n}
                    onSelectVOD={this.handleSelectVOD}
                    windowObj={windowObj}
                    playerHeight={playerDimensions.height}
                    playerWidth={playerDimensions.width}
                    onExit={this.handleExitRecommendations}
                    trackEvent={trackEvent}
                    streamWasOnline={streamHasPlayed}
                />
            );
        default:
            return null;
        }
    }

    _preloadVideoThumbnails() {
        /* Creating Images with video thumbnail sources makes the browser fetch the image
         * and prime the browser cache for quicker retrieval later.
         */
        this.props.recommendedVideos.forEach(video => {
            const image = new Image();
            image.src = video.thumbnailURL;
        });
    }

    handleSelectVOD(video) {
        this.props.selectRecommendedVideo(video.id);
        this.handleExitRecommendations();
    }

    handleExitRecommendations() {
        this.props.popScreen();
        this.props.clearRecommendedVODs();
    }
}

OfflineRecommendationsContainer.propTypes = propTypes;

// eslint-disable-next-line max-len
export const OfflineRecommendationsOverlay = connect(mapStateToProps, mapDispatchToProps)(OfflineRecommendationsContainer);
