import isEqual from 'lodash/isEqual';
import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { VOD_RECOMMENDATION_SCREEN, popScreen, pushScreen } from 'actions/screen';
import {
    clearRecommendedVODs,
    POST_VOD_RECOMMENDATIONS_TYPE,
    UNFETCHED,
    FETCHED,
    fetchRecommendedVODs,
    FETCH_VODS_THRESHOLD,
} from 'actions/recommendations';
import { selectRecommendedVideo } from 'actions/playback';
import includes from 'lodash/includes';
import { TRANSITION_TYPE_RECOMMENDATIONS } from 'state/playback';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import {
    PLAYER_CREATIVE,
    PLAYER_DASHBOARD,
    PLAYER_FEED,
    PLAYER_FRONTPAGE,
    PLAYER_HIGHLIGHTER,
} from 'util/player-type';
import { PostVODRecommendations } from 'ui/components/recommendations/post-vod-recommendations';

// Do not show recommendations on these player types
const INVALID_PLAYER_TYPES = [
    PLAYER_DASHBOARD,
    PLAYER_FRONTPAGE,
    PLAYER_FEED,
    PLAYER_CREATIVE,
    PLAYER_HIGHLIGHTER,
];

const propTypes = {
    hasVideoEnded: PropTypes.bool.isRequired,
    recommendedVideos: PropTypes.array.isRequired,
    windowObj: PropTypes.object.isRequired,
    selectRecommendedVideo: PropTypes.func.isRequired,
    popScreen: PropTypes.func.isRequired,
    clearRecommendedVODs: PropTypes.func.isRequired,
    i18n: PropTypes.object.isRequired,
    playerDimensions: PropTypes.shape({
        height: PropTypes.number,
        width: PropTypes.number,
    }).isRequired,
    trackEvent: PropTypes.func.isRequired,
    isMiniPlayer: PropTypes.bool.isRequired,
    shouldShowRecommendations: PropTypes.bool.isRequired,
    pushVODRecScreen: PropTypes.func.isRequired,
    fetchRecommendedVODs: PropTypes.func.isRequired,
    shouldFetchRecommendations: PropTypes.bool.isRequired,
    screen: PropTypes.array.isRequired,
};

// eslint-disable-next-line complexity
export const mapStateToProps = state => {
    const {
        recommendations,
        playback,
        lang,
        screen,
        window: windowObj,
        playerDimensions,
        analyticsTracker,
        ui,
        stream,
        env,
    } = state;

    const shouldShowRecommendations = (
        stream.contentType === CONTENT_MODE_VOD &&
        playback.ended &&
        !ui.isMini &&
        !includes(INVALID_PLAYER_TYPES, env.playerType) &&
        recommendations.status === FETCHED &&
        playback.transitionScheme === TRANSITION_TYPE_RECOMMENDATIONS &&
        recommendations.type === POST_VOD_RECOMMENDATIONS_TYPE
    );

    const shouldFetchRecommendations = (
        playback.transitionScheme === TRANSITION_TYPE_RECOMMENDATIONS &&
        stream.contentType === CONTENT_MODE_VOD &&
        (playback.duration - playback.currentTime) * 1000 < FETCH_VODS_THRESHOLD &&
        recommendations.status === UNFETCHED
    );

    return {
        hasVideoEnded: playback.ended,
        recommendedVideos: recommendations.videos,
        windowObj,
        i18n: lang,
        playerDimensions,
        trackEvent: analyticsTracker.trackEvent,
        isMiniPlayer: ui.isMini,
        shouldShowRecommendations: shouldShowRecommendations,
        shouldFetchRecommendations: shouldFetchRecommendations,
        screen,
    };
};

export const mapDispatchToProps = dispatch => ({
    fetchRecommendedVODs() {
        dispatch(fetchRecommendedVODs(POST_VOD_RECOMMENDATIONS_TYPE));
    },
    pushVODRecScreen() {
        dispatch(pushScreen(VOD_RECOMMENDATION_SCREEN));
    },
    selectRecommendedVideo(vodID) {
        dispatch(selectRecommendedVideo(vodID));
    },

    popScreen() {
        dispatch(popScreen());
    },

    clearRecommendedVODs() {
        dispatch(clearRecommendedVODs());
    },
});

export class VodPostplayRecommendationsContainer extends Component {
    constructor() {
        super(...arguments);
        this.handleSelectVOD = this.handleSelectVOD.bind(this);
        this.handleExitRecommendations = this.handleExitRecommendations.bind(this);

        this.state = {
            isImagePreloadingComplete: false,
        };
    }

    componentWillMount() {
        if (
            this.props.hasVideoEnded &&
            this.props.shouldShowRecommendations &&
            this.props.recommendedVideos.length > 0
        ) {
            this.props.pushVODRecScreen();
        }

        if (this.props.shouldFetchRecommendations) {
            this.props.fetchRecommendedVODs();
        }
    }

    // eslint-disable-next-line complexity
    componentWillReceiveProps(nextProps) {
        if (isEqual(this.props, nextProps)) {
            return;
        }

        const recommendedVideosChanged = nextProps.recommendedVideos !== this.props.recommendedVideos;
        const shouldPushRecommendations = (
            nextProps.screen[0] !== VOD_RECOMMENDATION_SCREEN &&
            nextProps.shouldShowRecommendations
        );
        const recommendationsWillLoadAndVideoEnded = (
            this.props.hasVideoEnded &&
            this.props.recommendedVideos.length === 0 &&
            nextProps.recommendedVideos.length > 0
        );
        const recommendationsLoadedAndVideoWillEnd  = (
            !this.props.hasVideoEnded &&
            nextProps.hasVideoEnded &&
            this.props.recommendedVideos.length > 0
        );
        if (
            recommendedVideosChanged &&
            nextProps.recommendedVideos.length > 0 &&
            this.state.isImagePreloadingComplete === false
        ) {
            this._preloadVideoThumbnails(nextProps.recommendedVideos);
        }

        // When recommendations just loaded and video already ended
        // or when video just ended and we have recommendations already

        if (
            (recommendationsWillLoadAndVideoEnded || recommendationsLoadedAndVideoWillEnd) &&
            shouldPushRecommendations
        ) {
            this.props.pushVODRecScreen();
            return;
        }

        if (nextProps.shouldFetchRecommendations) {
            this.props.fetchRecommendedVODs();
            return;
        }

        if (!nextProps.shouldShowRecommendations && nextProps.screen[0] === VOD_RECOMMENDATION_SCREEN) {
            this.props.popScreen();
            return;
        }
    }

    render() {
        const {
            recommendedVideos,
            windowObj,
            i18n,
            playerDimensions,
            trackEvent,
            isMiniPlayer,
            screen,
        } = this.props;

        if (screen[0] !== VOD_RECOMMENDATION_SCREEN) {
            return null;
        }

        return (
            <PostVODRecommendations
                videos={recommendedVideos}
                i18n={i18n}
                onSelectVOD={this.handleSelectVOD}
                windowObj={windowObj}
                playerHeight={playerDimensions.height}
                playerWidth={playerDimensions.width}
                onExit={this.handleExitRecommendations}
                trackEvent={trackEvent}
                isMiniPlayer={isMiniPlayer}
            />
        );
    }

    _preloadVideoThumbnails(videos) {
        /* Creating Images with video thumbnail sources makes the browser fetch the image
         * and prime the browser cache for quicker retrieval later.
         */
        videos.forEach(video => {
            const image = new Image();
            image.src = video.thumbnailURL;
        });

        this.setState({
            isImagePreloadingComplete: true,
        });
    }

    handleSelectVOD(video) {
        this.props.selectRecommendedVideo(video.id);
        this.handleExitRecommendations();
    }

    handleExitRecommendations() {
        // We do not need to call clearRecommendedVODs here
        // As recommendations are cleared when user switches videos or streams.
        // Also we want users to see recommendations again if current video seeks and user plays til end
        this.props.popScreen();
    }
}

VodPostplayRecommendationsContainer.propTypes = propTypes;

export const VodPostplayRecommendations =
    connect(mapStateToProps, mapDispatchToProps)(VodPostplayRecommendationsContainer);
