import React from 'react';
import PropTypes from 'prop-types';
import bowser from 'bowser';
import ReactDOM from 'react-dom';
import includes from 'lodash/includes';
import { connect } from 'react-redux';
import { MutedSegments } from '../components/seekbar/muted-segments';
import { SeekbarBuffer } from '../components/seekbar/seekbar-buffer';
import { SeekbarTimeDisplay } from '../components/seekbar/seekbar-time-display';
import { ClipsSeekbarTimeDisplay } from 'ui/player-types/clips/common/clips-seekbar-time-display';
import { Slider } from '../components/slider';
import { setLoading } from '../../actions/playback';
import { seek } from 'actions/video-api';
import { ThumbnailPreviews } from '../components/seekbar/thumbnail-previews';
import { TimestampPreview } from '../components/seekbar/timestamp-preview';
import { SeekbarMarkers } from '../components/seekbar/seekbar-markers';
import { SEEKABLE_TYPES } from 'util/seekable-types';
import { CLIPS_PLAYER_TYPES } from 'util/player-type';

const propTypes = {
    buffer: PropTypes.shape({
        start: PropTypes.number,
        end: PropTypes.number,
    }),
    currentTime: PropTypes.number,
    duration: PropTypes.number,
    getCurrentTime: PropTypes.func,
    isSeekableStream: PropTypes.bool.isRequired,
    mutedSegments: PropTypes.arrayOf(PropTypes.shape({
        duration: PropTypes.number,
        offset: PropTypes.number,
    })),
    seek: PropTypes.func.isRequired,
    seekbarMarkers: PropTypes.array,
    thumbnailPreviews: PropTypes.object,
    isClipsPlayerType: PropTypes.bool.isRequired,
    windowObj: PropTypes.object,
    trackEvent: PropTypes.func.isRequired,
};

const mapStateToProps = ({ env, playback, stream, timelineMetadata, window: windowObj, analyticsTracker }) => ({
    buffer: playback.buffer,
    currentTime: playback.currentTime,
    duration: playback.duration,
    isSeekableStream: includes(SEEKABLE_TYPES, stream.contentType),
    mutedSegments: timelineMetadata.mutedSegments,
    seekbarMarkers: timelineMetadata.markers,
    thumbnailPreviews: timelineMetadata.previews,
    windowObj: windowObj,
    trackEvent: analyticsTracker.trackEvent,
    isClipsPlayerType: CLIPS_PLAYER_TYPES.includes(env.playerType),
});

export const mapDispatchToProps = dispatch => ({
    seek(seekTime) {
        dispatch(seek(seekTime));
        dispatch(setLoading(true));
    },
});

export const classNames = {
    slider: 'player-slider player-slider--roundhandle js-player-slider',
    sliderLeft: 'ui-slider-range',
    sliderThumb: 'ui-slider-handle',
};

export const ANIMATION_SKIP_THRESHOLD = 0.05;

export class SeekbarComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            isDragging: false,
            isShowingMarkerPreview: false,
            mouseMoveClientX: 0,
            mouseMoveOnSeekbar: false,
            hideThumbnailPreview: false,
        };

        this.$seekbar = null;

        this.handleDragStart = this.handleDragStart.bind(this);
        this.handleDragStop = this.handleDragStop.bind(this);
        this.handleClick = this.handleClick.bind(this);
        this.handleMouseMove = this.handleMouseMove.bind(this);
        this.handleMouseOut = this.handleMouseOut.bind(this);
        this.seekbarRefHandler = this.seekbarRefHandler.bind(this);
        this.seekToPosition = this.seekToPosition.bind(this);
        this.handleShowingMarkerPreview = this.handleShowingMarkerPreview.bind(this);
        this.handleHidingMarkerPreview = this.handleHidingMarkerPreview.bind(this);
    }

    /* For some reason, eslint thinks render has a complexity of 6,
     * but as far as I can tell, it's only at 2. Disabling the rule as a result.
     */
    // eslint-disable-next-line complexity
    render() {
        const {
            currentTime,
            duration,
            isSeekableStream,
            isClipsPlayerType,
        } = this.props;

        if (!isSeekableStream) {
            return null;
        }

        const sliderComponent = this.createSliderComponent();

        const seekbarProps = {
            currentTime: currentTime,
            duration: duration,
        };

        return (
            <div className="player-seek">
                {isClipsPlayerType ?
                    <ClipsSeekbarTimeDisplay {...seekbarProps} /> :
                    <SeekbarTimeDisplay {...seekbarProps} />
                }
                {sliderComponent}
            </div>
        );
    }

    createSliderComponent() {
        const {
            isDragging,
            isShowingMarkerPreview,
            mouseMoveOnSeekbar,
        } = this.state;

        const {
            buffer,
            currentTime,
            duration,
            getCurrentTime,
            mutedSegments,
            seekbarMarkers,
            trackEvent,
            windowObj,
        } = this.props;

        const { seekbarWidth } = this.getSeekbarDimensions();

        const seekbarBufferComponent = isDragging ? null : (
            <SeekbarBuffer
                start={buffer.start}
                end={buffer.end}
                max={duration}
                min={0}
            />
        );

        const mutedSegmentsComponent = (
            <MutedSegments
                duration={duration}
                mutedSegments={mutedSegments}
            />
        );

        let seekbarPreviewsComponent = null;

        if (mouseMoveOnSeekbar && !isShowingMarkerPreview && duration > 0) {
            seekbarPreviewsComponent = this.createSeekbarPreviews();
        }

        const seekbarMarkersComponent = (
            <SeekbarMarkers
                currentTime={currentTime}
                duration={duration}
                markers={seekbarMarkers}
                onHidingMarkerPreview={this.handleHidingMarkerPreview}
                onMarkerSeek={this.seekToPosition}
                onShowingMarkerPreview={this.handleShowingMarkerPreview}
                seekbarWidth={seekbarWidth}
                trackEvent={trackEvent}
                hideThumbnailPreview= {this.state.hideThumbnailPreview}
            />
        );

        return (
            <Slider
                classNames={classNames}
                max={duration}
                min={0}
                onClick={this.handleClick}
                onMouseMove={this.handleMouseMove}
                onMouseOut={this.handleMouseOut}
                onBlur={this.handleMouseOut}
                ref={this.seekbarRefHandler}
                dragHandlers={{
                    onStart: this.handleDragStart,
                    onStop: this.handleDragStop,
                }}
                windowObj={windowObj}
                getOptimizedValue={getCurrentTime}
                valueOptimizationEnabled={true}
                behindSliderChildren={seekbarBufferComponent}
                afterSliderChildren={[
                    mutedSegmentsComponent,
                    seekbarPreviewsComponent,
                    seekbarMarkersComponent,
                ]}
            />
        );
    }

    getSeekbarDimensions() {
        let seekbarWidth = 0;
        let seekbarLeftOffset = 0;

        if (this.$seekbar) {
            const seekbarClientRect = this.$seekbar.getBoundingClientRect();
            seekbarWidth = seekbarClientRect.width;
            seekbarLeftOffset = seekbarClientRect.left;
        }
        return {
            seekbarWidth,
            seekbarLeftOffset,
        };
    }

    createSeekbarPreviews() {
        const { mouseMoveClientX } = this.state;
        const { duration, thumbnailPreviews, isClipsPlayerType } = this.props;
        const { seekbarWidth, seekbarLeftOffset } = this.getSeekbarDimensions();

        const isUnsupportedBrowser = bowser.msedge || bowser.msie;
        const showThumbnails = ((thumbnailPreviews.count > 0) && !isUnsupportedBrowser);

        return showThumbnails ? (
            <ThumbnailPreviews
                hideThumbnailPreview= {this.state.hideThumbnailPreview}
                duration={duration}
                previews={thumbnailPreviews}
                mouseMoveClientX={mouseMoveClientX}
                seekbarLeftOffset={seekbarLeftOffset}
                seekbarWidth={seekbarWidth}
            />
        ) : (
            <TimestampPreview
                duration={duration}
                mouseMoveClientX={mouseMoveClientX}
                seekbarLeftOffset={seekbarLeftOffset}
                seekbarWidth={seekbarWidth}
                shouldCleanString={isClipsPlayerType}
            />
        );
    }

    seekbarRefHandler(seekbar) {
        /* Reason for disabling:
         * We need to know the seekbar width/left when rendering previews/markers.
         * Unfortunately, the ref callback on a custom component returns the instance of
         * the component, which does not have a `getBoundingClientRect` method. So, we
         * have to use `findDOMNode` to get the actual dom node for the width/left values.
         */
        // eslint-disable-next-line react/no-find-dom-node
        this.$seekbar = ReactDOM.findDOMNode(seekbar);
    }

    handleClick(time) {
        /* Failing this condition means isDragging is false which shows
         * seekToPosition is already fired in handleDragStop
         */
        if (this.state.isDragging)
            this.seekToPosition(time);
    }

    handleMouseMove(e) {
        this.setState({
            mouseMoveOnSeekbar: true,
            mouseMoveClientX: e.clientX,
        });
    }

    handleMouseOut() {
        this.setState({ hideThumbnailPreview: true });
        const HIDE_THUMBNAIL_PREVIEW_TIMEOUT = 750;
        setTimeout(() => {
            if (this.state.hideThumbnailPreview) {
                this.setState({
                    mouseMoveOnSeekbar: false,
                    mouseMoveClientX: 0,
                    hideThumbnailPreview: false,
                });
            }
        }, HIDE_THUMBNAIL_PREVIEW_TIMEOUT);
    }

    handleDragStart() {
        this.setState({
            isDragging: true,
        });
    }

    handleDragStop(time) {
        this.setState({
            isDragging: false,
        });
        this.seekToPosition(time);
    }

    handleShowingMarkerPreview() {
        this.setState({
            isShowingMarkerPreview: true,
        });
    }

    handleHidingMarkerPreview() {
        this.setState({
            isShowingMarkerPreview: false,
        });
    }

    seekToPosition(time, markerId) {
        const marker = markerId ? { markerId: markerId } : {};
        this.props.seek(time);
        this.props.trackEvent('player_click_vod_seek', marker);
    }
}

SeekbarComponent.propTypes = propTypes;

export const Seekbar = connect(mapStateToProps, mapDispatchToProps)(SeekbarComponent);
