import React from 'react';
import PropTypes from 'prop-types';
import enhanceWithClickOutside from 'react-click-outside';
import { connect } from 'react-redux';
import { SettingsBanner } from 'ui/containers/settings/settings-banner';
import { SettingsButton } from '../components/settings/settings-button';
import { MenuManager } from '../components/settings/menu-manager';
import { showMenu, MENU_STATE_NONE, MENU_STATE_SETTINGS } from 'actions/ui';
import { QUALITY_AUTO } from 'actions/quality';
import { CLIPS_PLAYER_TYPES } from 'util/player-type';

const propTypes = {
    isAutoQuality: PropTypes.bool.isRequired,
    isEnded: PropTypes.bool.isRequired,
    isPaused: PropTypes.bool.isRequired,
    isSettingsShown: PropTypes.bool.isRequired,
    isClipsPlayerType: PropTypes.bool.isRequired,
    onShowSettings: PropTypes.func,
    playerHeight: PropTypes.number.isRequired,
    qualitiesAvailable: PropTypes.bool.isRequired,
    qualityIsChanging: PropTypes.bool.isRequired,
    trackEvent: PropTypes.func.isRequired,
    windowObj: PropTypes.object.isRequired,
};

const defaultProps = {
    showSettings() {},
};

export const mapStateToProps = ({ analyticsTracker, quality, playback, playerDimensions, ui, window, env }) => ({
    isAutoQuality: quality.selected === QUALITY_AUTO,
    isEnded: playback.ended,
    isPaused: playback.paused,
    isSettingsShown: ui.showSettings === MENU_STATE_SETTINGS,
    isClipsPlayerType: CLIPS_PLAYER_TYPES.includes(env.playerType),
    playerHeight: playerDimensions.height,
    qualitiesAvailable: quality.available.length > 0,
    qualityIsChanging: quality.selected !== quality.current,
    trackEvent: analyticsTracker.trackEvent,
    windowObj: window,
});

const mapDispatchToProps = dispatch => ({
    onShowSettings(show = MENU_STATE_NONE) {
        dispatch(showMenu(show));
    },
});

export class SettingsContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.handleSettingsButtonClick = this.handleSettingsButtonClick.bind(this);
        this.handleWindowBlur = this.handleWindowBlur.bind(this);
    }

    componentDidMount() {
        const { windowObj } = this.props;
        windowObj.addEventListener('blur', this.handleWindowBlur);
    }

    componentWillUnmount() {
        const { windowObj } = this.props;
        windowObj.removeEventListener('blur', this.handleWindowBlur);
    }

    handleWindowBlur() {
        if (this.props.isSettingsShown) {
            this.props.onShowSettings(MENU_STATE_NONE);
        }
    }

    handleClickOutside() {
        if (this.props.isSettingsShown) {
            this.props.onShowSettings(MENU_STATE_NONE);
        }
    }

    handleSettingsButtonClick() {
        const newMenuState = this.props.isSettingsShown ? MENU_STATE_NONE : MENU_STATE_SETTINGS;
        this.props.onShowSettings(newMenuState);
        this.props.trackEvent('player_click_options');
    }

    render() {
        const {
            isAutoQuality,
            isEnded,
            isPaused,
            isSettingsShown,
            playerHeight,
            qualitiesAvailable,
            qualityIsChanging,
            isClipsPlayerType,
        } = this.props;

        const menuManagerComponent = isSettingsShown ? (
            <MenuManager
                playerHeight={playerHeight}
            />
        ) : null;

        const qualityWillChange = !isAutoQuality && qualitiesAvailable && qualityIsChanging && !isClipsPlayerType;
        const isPlaying = !isEnded && !isPaused;
        const showSpinningAnimation = qualityWillChange && isPlaying;

        return (
            <div className="pl-flex">
                <SettingsBanner />
                <SettingsButton
                    onClick={this.handleSettingsButtonClick}
                    showSpinningAnimation={showSpinningAnimation}
                />
                {menuManagerComponent}
            </div>
        );
    }
}

SettingsContainer.propTypes = propTypes;
SettingsContainer.defaultProps = defaultProps;

export const Settings = connect(mapStateToProps, mapDispatchToProps)(enhanceWithClickOutside(SettingsContainer));
