import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { AdvancedMenu as AdvancedMenuComponent } from 'ui/components/settings/advanced-menu';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { PLAYER_CURSE, PLAYER_SITE } from 'util/player-type';
import { displayStats } from 'actions/stats';
import { enablePlayerPersistence, disablePlayerPersistence } from 'actions/settings';
import { changeBackend } from 'actions/video-api';
import { BACKEND_FLASH } from 'backend/flash';
import { Experiment } from 'ui/containers/common/experiments-container';
import { HTML5_TOGGLE_REMOVAL } from 'experiments';
import { getAvailableBackends } from 'backend/util';

const propTypes = {
    availableBackends: PropTypes.array.isRequired,
    isCursePlayer: PropTypes.bool.isRequired,
    isHTML5Player: PropTypes.bool.isRequired,
    isMiniPlayerEnabled: PropTypes.bool.isRequired,
    isSitePlayer: PropTypes.bool.isRequired,
    onMenuTransition: PropTypes.func,
    statsDisplayed: PropTypes.bool.isRequired,
    toggleHTML5Player: PropTypes.func.isRequired,
    toggleMiniPlayer: PropTypes.func.isRequired,
    toggleVideoStats: PropTypes.func.isRequired,
};

const defaultProps = {
    onMenuTransition() {},
};

export const mapStateToProps = ({ backendInfo, env, settings, stats }) => {
    const backendType = backendInfo.playerBackendType;
    return {
        isCursePlayer: env.playerType === PLAYER_CURSE,
        isHTML5Player: backendType === BACKEND_PLAYER_CORE || backendType === BACKEND_MEDIA_PLAYER,
        isMiniPlayerEnabled: settings.persistenceEnabled,
        isSitePlayer: env.playerType === PLAYER_SITE,
        statsDisplayed: stats.displayed,
        availableBackends: getAvailableBackends(env.playerType),
    };
};

export const mapDispatchToProps = dispatch => ({
    toggleVideoStats(enabled = false) {
        if (enabled) {
            dispatch(displayStats(true));
        } else {
            dispatch(displayStats(false));
        }
    },
    toggleMiniPlayer(enabled = false) {
        if (enabled) {
            dispatch(enablePlayerPersistence());
        } else {
            dispatch(disablePlayerPersistence());
        }
    },
    toggleHTML5Player(enabled = false) {
        if (enabled) {
            dispatch(changeBackend(BACKEND_PLAYER_CORE));
        } else {
            dispatch(changeBackend(BACKEND_FLASH));
        }
    },
});

export class AdvancedMenuContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.toggleVideoStats = this.toggleVideoStats.bind(this);
        this.toggleMiniPlayer = this.toggleMiniPlayer.bind(this);
        this.toggleHTML5Player = this.toggleHTML5Player.bind(this);
    }

    toggleVideoStats() {
        this.props.toggleVideoStats(!this.props.statsDisplayed);
    }

    toggleMiniPlayer() {
        this.props.toggleMiniPlayer(!this.props.isMiniPlayerEnabled);
    }

    toggleHTML5Player() {
        this.props.toggleHTML5Player(!this.props.isHTML5Player);
    }

    render() {
        const {
            availableBackends,
            statsDisplayed,
            isCursePlayer,
            isHTML5Player,
            isMiniPlayerEnabled,
            isSitePlayer,
            onMenuTransition,
        } = this.props;

        const canPlayHtml5 = availableBackends.some(backend => {
            return backend === BACKEND_PLAYER_CORE || backend === BACKEND_MEDIA_PLAYER;
        });

        const canPlayFlash = availableBackends.some(backend => {
            return backend === BACKEND_FLASH;
        });

        const html5ToggleEnabled = (
            !isCursePlayer &&
            canPlayHtml5 &&
            canPlayFlash
        );

        return (
            <div>
                <Experiment
                    uuid={HTML5_TOGGLE_REMOVAL}
                    renderValue={'keep'}
                >
                    <AdvancedMenuComponent
                        html5PlayerEnabled={isHTML5Player}
                        html5ToggleEnabled={html5ToggleEnabled}
                        isShowingVideoStats={statsDisplayed}
                        miniPlayerEnabled={isMiniPlayerEnabled}
                        miniPlayerToggleEnabled={isSitePlayer}
                        onMenuTransition={onMenuTransition}
                        onMiniPlayerToggle={this.toggleMiniPlayer}
                        onStatsToggle={this.toggleVideoStats}
                        onHTML5Toggle={this.toggleHTML5Player}
                    />
                </Experiment>
                <Experiment
                    uuid={HTML5_TOGGLE_REMOVAL}
                    renderValue={'remove'}
                >
                    <AdvancedMenuComponent
                        html5PlayerEnabled={isHTML5Player}
                        html5ToggleEnabled={false}
                        isShowingVideoStats={statsDisplayed}
                        miniPlayerEnabled={isMiniPlayerEnabled}
                        miniPlayerToggleEnabled={isSitePlayer}
                        onMenuTransition={onMenuTransition}
                        onMiniPlayerToggle={this.toggleMiniPlayer}
                        onStatsToggle={this.toggleVideoStats}
                        onHTML5Toggle={this.toggleHTML5Player}
                    />
                </Experiment>
            </div>
        );
    }
}

AdvancedMenuContainer.propTypes = propTypes;
AdvancedMenuContainer.defaultProps = defaultProps;

export const AdvancedMenu = connect(mapStateToProps, mapDispatchToProps)(AdvancedMenuContainer);
