import React from 'react';
import PropTypes from 'prop-types';
import find from 'lodash/find';
import includes from 'lodash/includes';
import { connect } from 'react-redux';
import { MainMenu as MainMenuComponent } from 'ui/components/settings/main-menu';
import { popoutPlayer } from 'util/popout-player';
import { pause } from 'actions/video-api';
import { toURLString as timestampUrlString } from 'util/timestamp';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_CLIP } from 'stream/clip';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { BACKEND_CLIP } from 'backend/clip';
import { showMenu, MENU_STATE_NONE } from 'actions/ui';
import { showCCModal } from 'actions/captions';
import { PLAYER_AMAZON_VSE, CLIPS_PLAYER_TYPES } from 'util/player-type';
import { pushScreen, CLIP_SHORTCUT_SCREEN } from 'actions/screen';

const PLAYBACK_SPEED_SUPPORTED_BACKENDS = [BACKEND_PLAYER_CORE, BACKEND_MEDIA_PLAYER, BACKEND_CLIP];

const propTypes = {
    availableQualities: PropTypes.array.isRequired,
    backendType: PropTypes.string.isRequired,
    captionsAvailable: PropTypes.bool.isRequired,
    collection: PropTypes.object.isRequired,
    currentSpeed: PropTypes.number.isRequired,
    currentTime: PropTypes.number,
    hideSettingsMenu: PropTypes.func,
    isStaffUser: PropTypes.bool,
    isVod: PropTypes.bool.isRequired,
    isClip: PropTypes.bool.isRequired,
    onMenuTransition: PropTypes.func,
    pausePlayback: PropTypes.func,
    playback: PropTypes.object.isRequired,
    playerType: PropTypes.string,
    selectedQuality: PropTypes.string.isRequired,
    showClipsShortcutOverlay: PropTypes.func.isRequired,
    showCaptionsOptions: PropTypes.func.isRequired,
    stream: PropTypes.object.isRequired,
    streamUrl: PropTypes.string,
    windowObj: PropTypes.object.isRequired,
};

const defaultProps = {
    currentTime: 0,
    isStaffUser: false,
    hideSettingsMenu() {},
    onMenuTransition() {},
    pausePlayback() {},
    streamUrl: '',
    playerType: '',
};

const mapStateToProps = ({
    backendInfo,
    captions,
    collection,
    env,
    playback,
    quality,
    stream,
    streamMetadata,
    user,
    window,
}) => ({
    availableQualities: quality.available,
    backendType: backendInfo.playerBackendType,
    captionsAvailable: captions.available,
    collection,
    currentSpeed: playback.playbackRate,
    currentTime: playback.currentTime,
    isStaffUser: user.isStaff,
    isVod: stream.contentType === CONTENT_MODE_VOD,
    isClip: stream.contentType === CONTENT_MODE_CLIP,
    playback,
    playerType: env.playerType,
    selectedQuality: quality.selected,
    stream,
    streamUrl: streamMetadata.url,
    windowObj: window,
});

export const mapDispatchToProps = dispatch => ({
    pausePlayback() {
        dispatch(pause());
    },
    hideSettingsMenu() {
        dispatch(showMenu(MENU_STATE_NONE));
    },
    showCaptionsOptions() {
        dispatch(showCCModal(true));
    },
    showClipsShortcutOverlay() {
        dispatch(pushScreen(CLIP_SHORTCUT_SCREEN));
    },
});

export class MainMenuContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.showCaptionsOptions = this.showCaptionsOptions.bind(this);
        this.popoutPlayer = this.popoutPlayer.bind(this);
    }

    showCaptionsOptions() {
        this.props.hideSettingsMenu();
        this.props.showCaptionsOptions();
    }

    popoutPlayer() {
        this.props.pausePlayback();
        this.props.hideSettingsMenu();

        popoutPlayer({
            collection: this.props.collection,
            playback: this.props.playback,
            stream: this.props.stream,
            window: this.props.windowObj,
            playerType: this.props.playerType,
        });
    }

    render() {
        const {
            availableQualities,
            backendType,
            captionsAvailable,
            currentTime,
            currentSpeed,
            isStaffUser,
            isVod,
            isClip,
            onMenuTransition,
            playerType,
            selectedQuality,
            streamUrl,
            showClipsShortcutOverlay,
            windowObj,
        } = this.props;

        const canChangePlaybackSpeed = (
            includes(PLAYBACK_SPEED_SUPPORTED_BACKENDS, backendType) ||
            includes(CLIPS_PLAYER_TYPES, playerType)
        );
        const canPopoutPlayer = playerType !== PLAYER_AMAZON_VSE;
        const selectedQualityObj = find(availableQualities, qualityObj => qualityObj.group === selectedQuality);
        const selectedQualityName = selectedQualityObj ? selectedQualityObj.name : '';
        const vodUrlWithTimestamp = `${streamUrl}?t=${timestampUrlString(currentTime)}`;
        const showQualityMenu = availableQualities.length > 0;

        return (
            <MainMenuComponent
                playerType={playerType}
                canChangePlaybackSpeed={canChangePlaybackSpeed}
                canPopoutPlayer={canPopoutPlayer}
                captionsAvailable={captionsAvailable}
                currentTime={currentTime}
                currentSpeed={currentSpeed}
                isStaffUser={isStaffUser}
                isVod={isVod}
                isClip={isClip}
                onMenuTransition={onMenuTransition}
                onPopoutPlayer={this.popoutPlayer}
                onShowCaptionsOptions={this.showCaptionsOptions}
                selectedQualityName={selectedQualityName}
                shouldShowQualityMenu={showQualityMenu}
                vodUrlWithTimestamp={vodUrlWithTimestamp}
                showClipsShortcutOverlay={showClipsShortcutOverlay}
                windowObj={windowObj}
            />
        );
    }
}

MainMenuContainer.propTypes = propTypes;
MainMenuContainer.defaultProps = defaultProps;

export const MainMenu = connect(mapStateToProps, mapDispatchToProps)(MainMenuContainer);
