import React from 'react';
import PropTypes from 'prop-types';
import find from 'lodash/find';
import { connect } from 'react-redux';
import { translate } from 'react-i18next';
import { SettingsBanner as SettingsBannerComponent } from 'ui/components/settings/settings-banner';

// The amount of time the banner should be shown for quality changes
export const QUALITY_CHANGE_DURATION = 5000;

const propTypes = {
    availableQualities: PropTypes.array.isRequired,
    currentQuality: PropTypes.string.isRequired,
    playbackRate: PropTypes.number.isRequired,
    t: PropTypes.func.isRequired,
    windowObj: PropTypes.object.isRequired,
};

const mapStateToProps = ({ playback, quality, window }) => ({
    availableQualities: quality.available,
    currentQuality: quality.current,
    playbackRate: playback.playbackRate,
    windowObj: window,
});

export class SettingsBannerContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            // This state property is used to ensure the banner needs to be rendered the first time it is
            // Otherwise, due to the default CSS state of `animateOut`, the banner flashes onscreen as it is mounted
            canRenderBanner: false,
            lastQuality: '',
            hasChangedQuality: false,
        };

        this.bannerHideTimeout = 0;
    }

    componentWillReceiveProps(nextProps) {
        if (nextProps.currentQuality !== this.state.lastQuality) {
            this._onQualityPropChange(nextProps);
        }

        if (
            this.state.hasChangedQuality ||
            nextProps.currentQuality !== this.state.lastQuality ||
            nextProps.playbackRate !== 1.0
        ) {
            this.setState({
                canRenderBanner: true,
            });
        }
    }

    componentWillUnmount() {
        if (this.bannerHideTimeout !== 0) {
            this.props.windowObj.clearTimeout(this.bannerHideTimeout);
        }
    }

    clearBannerHideTimeout() {
        this.props.windowObj.clearTimeout(this.bannerHideTimeout);
        this.bannerHideTimeout = 0;
    }

    showBannerForQuality(nextQuality) {
        this.setState({
            lastQuality: nextQuality,
            hasChangedQuality: true,
        });
    }

    hideBannerForQuality() {
        this.setState({
            hasChangedQuality: false,
        });
        this.bannerHideTimeout = 0;
    }

    _onQualityPropChange(nextProps) {
        if (this.bannerHideTimeout !== 0) {
            this.clearBannerHideTimeout();
        }

        this.showBannerForQuality(nextProps.currentQuality);

        this.bannerHideTimeout = nextProps.windowObj.setTimeout(() => {
            this.hideBannerForQuality();
        }, QUALITY_CHANGE_DURATION);
    }

    render() {
        const { playbackRate } = this.props;
        const { canRenderBanner, hasChangedQuality } = this.state;

        const shouldShowBanner = hasChangedQuality || playbackRate !== 1.0;
        const bannerLabel = this._getBannerLabel();

        return canRenderBanner ? (
            <SettingsBannerComponent
                bannerLabel={bannerLabel}
                showBanner={shouldShowBanner}
            />
        ) : null;
    }

    _getBannerLabel() {
        const {
            playbackRate,
        } = this.props;

        if (this.state.hasChangedQuality) {
            return this._getQualityName();
        }

        if (playbackRate !== 1.0) {
            return `${playbackRate}x`;
        }

        return '';
    }

    _getQualityName() {
        const { availableQualities, currentQuality, t } = this.props;
        const currentQualityObj = find(availableQualities, qualityObj => qualityObj.group === currentQuality);
        const qualityName = currentQualityObj ? currentQualityObj.name : '';
        return t(qualityName);
    }
}

SettingsBannerContainer.propTypes = propTypes;

export const SettingsBanner = translate()(connect(mapStateToProps)(SettingsBannerContainer));
