import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { LinkedImage } from '../../components/common/linked-image';
import { StreamInfoDisplay } from '../../components/stream-info/stream-info-display';
import { gamePath } from '../../../settings';
import { pause } from '../../../actions/video-api';
import { PLAYER_TWITCH_EVERYWHERE, PLAYER_IMDB } from 'util/player-type';
import { TwitchEverywherePanelContainer } from 'ui/containers/twitch-everywhere-panel';
import { WatchPartyStreamInfo } from 'ui/components/stream-info/watch-party';
import { AdContentTypes } from 'actions/ads';
import classNames from 'classnames';
import { SIDEBAR_VIEW } from 'state/collection';
import { PLATFORM_MOBILE_WEB } from 'state/env';

const propTypes = {
    adPlaying: PropTypes.bool,
    channelName: PropTypes.string,
    displayName: PropTypes.string,
    gameName: PropTypes.string,
    isWatchParty: PropTypes.bool,
    lang: PropTypes.object,
    logo: PropTypes.string,
    name: PropTypes.string,
    pause: PropTypes.func,
    showInfo: PropTypes.bool,
    status: PropTypes.string,
    streamUrl: PropTypes.string,
    playerType: PropTypes.string,
    isMobile: PropTypes.bool,
    viewerCount: PropTypes.number,
    vodId: PropTypes.string,
    vodTitle: PropTypes.string,
    vodIsViewable: PropTypes.bool,
    vodBroadcastType: PropTypes.string,
};

export const mapStateToProps = ({
    ads, playerOptions, env, lang,
    screenMode, streamMetadata, viewercount, watchParty,
    collection,
}) => ({
    adPlaying: (ads.currentMetadata.contentType !== AdContentTypes.NONE),
    channelName: streamMetadata.channelName,
    displayName: streamMetadata.channel.displayName,
    gameName: streamMetadata.game,
    lang: lang,
    logo: streamMetadata.channel.logo,
    name: streamMetadata.name,
    showInfo: (
        playerOptions.showInfo ||
        screenMode.isFullScreen ||
        screenMode.isTheatreMode
    ) && collection.currentView !== SIDEBAR_VIEW,
    status: streamMetadata.channel.status,
    streamUrl: streamMetadata.url,
    viewerCount: viewercount,
    playerType: env.playerType,
    isMobile: env.platform === PLATFORM_MOBILE_WEB,
    isWatchParty: streamMetadata.streamType === 'watch_party',
    vodId: watchParty.vodId,
    vodTitle: watchParty.title,
    vodIsViewable: watchParty.viewable === 'public',
    vodBroadcastType: watchParty.broadcastType,
});

export const mapDispatchToProps = dispatch => ({
    pause() {
        dispatch(pause());
    },
});

const LOGO_CLASSES = {
    a: 'player-streaminfo__picture',
    img: 'qa-broadcaster-logo',
};

export class StreamInfoComponent extends React.PureComponent {
    render() {
        const {
            adPlaying,
            playerType,
            showInfo,
        } = this.props;

        if (adPlaying || playerType === PLAYER_IMDB) {
            return null;
        }

        const isTwitchEverywhere = this.props.playerType === PLAYER_TWITCH_EVERYWHERE;
        const teFollowPanel = this._createTEFollowPanel(isTwitchEverywhere);
        const logoComponent = this._createLogoComponent(isTwitchEverywhere);
        const watchPartyStreamInfo = this.props.isWatchParty ? <WatchPartyStreamInfo {...this.props} /> : null;

        return showInfo ? (
            <div className="player-streaminfo">
                {watchPartyStreamInfo}
                {logoComponent}
                {this._createStreamInfoDisplay()}
                {teFollowPanel}
            </div>
        ) : null;
    }

    _createTEFollowPanel(isTwitchEverywhere) {
        return isTwitchEverywhere ? <TwitchEverywherePanelContainer /> : null;
    }

    _createLogoComponent(isTwitchEverywhere) {
        const {
            logo,
            streamUrl,
        } = this.props;

        const target = '_blank';

        return (!isTwitchEverywhere && logo) ? (
            <LinkedImage
                classNames={LOGO_CLASSES}
                imgSrc={logo}
                href={streamUrl}
                onClick={this.props.pause}
                target={target}
            />
        ) : null;
    }

    _createStreamInfoDisplay() {
        const {
            displayName,
            channelName,
            name,
            status,
            gameName,
            streamUrl,
            viewerCount,
            playerType,
            isMobile,
            lang,
        } = this.props;

        // use status if we're live, use name for vod
        const live = Boolean(channelName);
        const title = live ? status : name;
        const gameUrl = `${gamePath}/${encodeURIComponent(gameName)}`;

        const isTwitchEverywhere = playerType === PLAYER_TWITCH_EVERYWHERE;

        const className = classNames({
            'player-streaminfo__meta-container': !isTwitchEverywhere,
        });

        return displayName ? (
            <StreamInfoDisplay
                className={className}
                live={live}
                name={displayName}
                title={title}
                gameName={gameName}
                gameUrl={gameUrl}
                onClick={this.props.pause}
                streamUrl={streamUrl}
                viewerCount={viewerCount}
                lang={lang}
                isMobile={isMobile}
            />
        ) : null;
    }
}

StreamInfoComponent.propTypes = propTypes;

export const StreamInfo = connect(mapStateToProps, mapDispatchToProps)(StreamInfoComponent);
