import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { IconComponent } from 'ui/components/buttons/base-icon';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { VOD_RECOMMENDATION_SCREEN } from 'actions/screen';
import { ONLINE_STATUS } from 'state/online-status';
import { TWILIGHT_PREMIERE_UPLOAD_FLOW } from 'experiments';
import { Experiment } from '../common/experiments-container';
import classNames from 'classnames';

const propTypes = {
    streamType: PropTypes.string,
    i18n: PropTypes.object.isRequired,
    isOnline: PropTypes.bool,
    inCollection: PropTypes.bool,
    isWatchParty: PropTypes.bool,
    isRerun: PropTypes.bool,
    isPremiere: PropTypes.bool,
    showingRecommendations: PropTypes.bool,
};

export const mapStateToProps = ({ lang, onlineStatus, stream, collection, streamMetadata, screen }) => ({
    streamType: stream.contentType,
    i18n: lang,
    isOnline: onlineStatus === ONLINE_STATUS,
    inCollection: Boolean(collection.id),
    // TODO [VWT-1119]: after premieres launch, remove `isWatchParty`
    isWatchParty: streamMetadata.streamType === 'watch_party',
    isRerun: streamMetadata.streamType === 'rerun',
    isPremiere: streamMetadata.streamType === 'premiere',
    showingRecommendations: screen[0] === VOD_RECOMMENDATION_SCREEN,
});

const LIVE_STATUS = 'Live';
const OFFLINE_STATUS = 'Offline';
// TODO [VWT-1119]: after premieres launch, remove `WATCH_PARTY_STATUS`
const WATCH_PARTY_STATUS = 'Vodcast';
const WATCH_PARTY_RERUN_STATUS = 'Rerun';
const WATCH_PARTY_PREMIERE_STATUS = 'Premiere';

const STREAMSTATUS_CLASSES = classNames(
    'player-streamstatus'
);

// Disabling complexity for this stateless component.
/* eslint-disable complexity */
export const StreamStatusComponent = ({
    streamType,
    i18n,
    isOnline,
    inCollection,
    isWatchParty,
    isRerun,
    isPremiere,
    showingRecommendations,
}) => {
    let statusComponent = null;
    if (
        !inCollection &&
        !showingRecommendations
    ) {
        const LIVE_STATUS_CLASSES = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--live',
            'qa-live-label'
        );
        const OFFLINE_STATUS_CLASSES = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--offline'
        );
        const WATCH_PARTY_STATUS_CLASSES = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--watchparty'
        );
        const WATCH_PARTY_RERUN_STATUS_CLASSES = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--rerun'
        );
        const WATCH_PARTY_PREMIERE_STATUS_CLASSES = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--premiere'
        );

        if (streamType === CONTENT_MODE_LIVE) {
            if (isWatchParty && isOnline) {
                statusComponent = (
                    <span>
                        <IconComponent
                            svgClass={WATCH_PARTY_STATUS_CLASSES}
                            iconId={'#icon_vodcast'}
                        />
                        <span className="player-streamstatus__label">
                            {i18n.translate(WATCH_PARTY_STATUS)}
                        </span>
                    </span>
                );
            } else if (isRerun && isOnline) {
                statusComponent = (
                    <span>
                        <Experiment uuid={TWILIGHT_PREMIERE_UPLOAD_FLOW} renderValue={'yes'}>
                            <span className="player-streamstatus__container">
                                <IconComponent
                                    svgClass={WATCH_PARTY_RERUN_STATUS_CLASSES}
                                    iconId={'#icon_rerun'}
                                />
                                <span className="player-streamstatus__label">
                                    {i18n.translate(WATCH_PARTY_RERUN_STATUS)}
                                </span>
                            </span>
                        </Experiment>
                        <Experiment uuid={TWILIGHT_PREMIERE_UPLOAD_FLOW} renderValue={'no'}>
                            <span>
                                <IconComponent
                                    svgClass={WATCH_PARTY_STATUS_CLASSES}
                                    iconId={'#icon_vodcast'}
                                />
                                <span className="player-streamstatus__label">
                                    {i18n.translate(WATCH_PARTY_STATUS)}
                                </span>
                            </span>
                        </Experiment>
                    </span>
                );
            } else if (isPremiere && isOnline) {
                statusComponent = (
                    <span>
                        <Experiment uuid={TWILIGHT_PREMIERE_UPLOAD_FLOW} renderValue={'yes'}>
                            <span className="player-streamstatus__container">
                                <IconComponent
                                    svgClass={WATCH_PARTY_PREMIERE_STATUS_CLASSES}
                                    iconId={'#icon_premiere'}
                                />
                                <span className="player-streamstatus__label">
                                    {i18n.translate(WATCH_PARTY_PREMIERE_STATUS)}
                                </span>
                            </span>
                        </Experiment>
                        <Experiment uuid={TWILIGHT_PREMIERE_UPLOAD_FLOW} renderValue={'no'}>
                            <span>
                                <IconComponent
                                    svgClass={WATCH_PARTY_STATUS_CLASSES}
                                    iconId={'#icon_vodcast'}
                                />
                                <span className="player-streamstatus__label">
                                    {i18n.translate(WATCH_PARTY_STATUS)}
                                </span>
                            </span>
                        </Experiment>
                    </span>
                );
            } else if (isOnline) {
                statusComponent = (
                    <span>
                        <IconComponent
                            svgClass={LIVE_STATUS_CLASSES}
                            iconId={'#icon_live'}
                        />
                        <span className="player-streamstatus__label">
                            {i18n.translate(LIVE_STATUS)}
                        </span>
                    </span>
                );
            } else {
                statusComponent = (
                    <span>
                        <IconComponent
                            svgClass={OFFLINE_STATUS_CLASSES}
                            iconId={'#icon_offline'}
                        />
                        <span className="player-streamstatus__label">
                            {i18n.translate(OFFLINE_STATUS)}
                        </span>
                    </span>
                );
            }
        }
    }

    return (
        <div className={STREAMSTATUS_CLASSES}>
            {statusComponent}
        </div>
    );
};

StreamStatusComponent.propTypes = propTypes;

export const StreamStatus = connect(mapStateToProps)(StreamStatusComponent);
