import React from 'react';
import PropTypes from 'prop-types';
import { NOTIFICATIONS_BUTTON_VIEW, followChannel, unfollowChannel } from 'actions/follow';
import { connect } from 'react-redux';
import { TwitchEverywhereFollowPanel } from 'ui/components/follow-panels/twitch-everywhere-panel';
import { LOGGED_IN } from 'state/user';
import { PLAYER_TWITCH_EVERYWHERE } from 'util/player-type';

export const AUTHENTICATION_REQUIRED = 'authentication-required';
export const SUBSCRIBE = 'subscribe';
export const TE_NAMESPACE = 'twitch-everywhere';

const propTypes = {
    avatarURL: PropTypes.string,
    channelName: PropTypes.string.isRequired,
    channelId: PropTypes.number.isRequired,
    userId: PropTypes.number.isRequired,
    isLoggedIn: PropTypes.bool.isRequired,
    isFollowing: PropTypes.bool.isRequired,
    didFetchFollow: PropTypes.bool.isRequired,
    didFetchSubscriptions: PropTypes.bool.isRequired,
    followChannel: PropTypes.func.isRequired,
    unfollowChannel: PropTypes.func.isRequired,
    window: PropTypes.object.isRequired,
    notificationsEnabled: PropTypes.bool.isRequired,
    analytics: PropTypes.object.isRequired,
    targetOrigin: PropTypes.string.isRequired,
};

const mapStateToProps = state => ({
    avatarURL: state.streamMetadata.channel.logo,
    channelName: state.streamMetadata.channel.name,
    channelId: state.streamMetadata.channel.id,
    userId: state.user.id,
    isLoggedIn: state.user.loggedInStatus === LOGGED_IN,
    isFollowing: state.follow.following,
    didFetchSubscriptions: state.subscriptions.didFetch,
    didFetchFollow: state.follow.fetched,
    window: state.window,
    notificationsEnabled: state.follow.notificationsEnabled,
    analytics: state.analyticsTracker,
    targetOrigin: state.env.twitchEverywhere.targetOrigin,
});

export const mapDispatchToProps = dispatch => ({
    followChannel(userId, channelId) {
        dispatch(followChannel(userId, channelId));
    },

    unfollowChannel(userId, channelId) {
        dispatch(unfollowChannel(userId, channelId));
    },
});

export class TwitchEverywherePanel extends React.Component {
    constructor() {
        super(...arguments);
        this.handleFollowChannel = this.handleFollowChannel.bind(this);
        this.handleUnfollowChannel = this.handleUnfollowChannel.bind(this);
        this.handleSubscribe = this.handleSubscribe.bind(this);
    }

    componentDidMount() {
        this.props.analytics.trackEvent(NOTIFICATIONS_BUTTON_VIEW, {
            /* eslint-disable camelcase */
            follow_state: this.props.isFollowing,
            notif_state: this.props.notificationsEnabled,
            /* eslint-enable */
            src: PLAYER_TWITCH_EVERYWHERE,
        });
    }

    postMessage(eventName, params = {}) {
        this.props.window.parent.postMessage({
            eventName,
            namespace: TE_NAMESPACE,
            params,
        }, this.props.targetOrigin);
    }

    handleFollowChannel() {
        if (this.props.isLoggedIn) {
            this.props.followChannel(this.props.userId, this.props.channelId);
        } else {
            this.postMessage(AUTHENTICATION_REQUIRED);
        }
    }

    handleUnfollowChannel() {
        this.props.unfollowChannel(this.props.userId, this.props.channelId);
    }

    handleSubscribe() {
        const { channelName } = this.props;

        this.postMessage(SUBSCRIBE, {
            channelName,
        });
    }

    render() {
        const isAnonymousWithChannel = !this.props.isLoggedIn && this.props.channelName;
        const didFetchAllData = this.props.didFetchFollow && this.props.didFetchSubscriptions;
        const isOwnChannel = this.props.channelId === this.props.userId;

        if (isAnonymousWithChannel || (didFetchAllData && !isOwnChannel)) {
            return (
                <TwitchEverywhereFollowPanel
                    avatarURL={this.props.avatarURL}
                    onFollowChannel={this.handleFollowChannel}
                    onUnfollowChannel={this.handleUnfollowChannel}
                    onSubscribe={this.handleSubscribe}
                />
            );
        }

        return null;
    }
}

TwitchEverywherePanel.propTypes = propTypes;

const withStoreProps = connect(mapStateToProps, mapDispatchToProps);
export const TwitchEverywherePanelContainer = withStoreProps(TwitchEverywherePanel);
