import React from 'react';
import PropTypes from 'prop-types';
import { translate } from 'react-i18next';
import { connect } from 'react-redux';
import { Slider } from '../components/slider';
import { changeVolume, mutePlayer } from '../../actions/video-api';
import { volumeStepAmount } from '../../settings';
import { trackEvent } from 'actions/analytics-tracker';
import classNames from 'classnames';
import { isPlayingSound } from 'state/playback';

const TRACK_EVENT_PLAYER_CLICK = 'player_click';
const TRACK_COMPONENT_VOLUME_BUTTON = 'volume_button';
const TRACK_COMPONENT_STATE_UNMUTE = 'unmute';
const TRACK_COMPONENT_STATE_MUTE = 'mute';

const propTypes = {
    muted: PropTypes.bool,
    mutePlayer: PropTypes.func,
    changeVolume: PropTypes.func,
    volume: PropTypes.number,
    trackVolumeButtonClick: PropTypes.func,
    isPlayingSound: PropTypes.bool,
    t: PropTypes.func,
    showVolumeSlider: PropTypes.bool,
};

export const mapStateToProps = state => ({
    muted: state.playback.muted,
    volume: state.playback.volume,
    isPlayingSound: isPlayingSound(state),
    showVolumeSlider: state.ui.showVolumeSlider,
});

export const mapDispatchToProps = dispatch => ({
    mutePlayer(muted) {
        dispatch(mutePlayer(muted, false));
    },

    changeVolume(value) {
        dispatch(changeVolume(value));
    },

    trackVolumeButtonClick(muting) {
        const componentState = muting ? TRACK_COMPONENT_STATE_MUTE : TRACK_COMPONENT_STATE_UNMUTE;
        dispatch(trackEvent(TRACK_EVENT_PLAYER_CLICK, {
            component: TRACK_COMPONENT_VOLUME_BUTTON,
            // eslint-disable-next-line camelcase
            component_state: componentState,
        }));
    },
});

export const sliderClassNames = {
    slider: 'player-volume__slider player-slider',
    sliderLeft: 'ui-slider-range',
    sliderThumb: 'ui-slider-handle',
};

const VOLUME_SLIDER_CONTAINER_CLASSES = classNames(
    'pl-flex',
    'pl-flex__item--grow',
    'player-volume'
);

export class VolumeSliderComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.handleIconClick = this.handleIconClick.bind(this);
        this.handleSliderChange = this.handleSliderChange.bind(this);
    }

    render() {
        const {
            muted,
            volume,
            showVolumeSlider,
        } = this.props;

        if (!showVolumeSlider) {
            return null;
        }

        const volumeValue = muted ? 0 : volume;
        const volumeIcon = this.getVolumeIcon();

        return (
            <div className={VOLUME_SLIDER_CONTAINER_CLASSES}>
                <button
                    type="button"
                    className="player-button player-button--volume qa-control-volume"
                    onClick={this.handleIconClick}
                >
                    {volumeIcon}
                </button>
                <div className="player-volume__slider-container">
                    <Slider
                        classNames={sliderClassNames}
                        max={1}
                        min={0}
                        value={volumeValue}
                        skipAnimation={true}
                        dragHandlers={{
                            onDrag: this.handleSliderChange,
                            onStop: this.handleSliderChange,
                        }}
                        onClick={this.handleSliderChange}
                    />
                </div>
            </div>
        );
    }

    handleSliderChange(value) {
        const { changeVolume, mutePlayer } = this.props;

        changeVolume(value);
        mutePlayer(false);
    }

    handleIconClick() {
        const {
            volume, changeVolume, mutePlayer,
            trackVolumeButtonClick, isPlayingSound,
        } = this.props;

        if (isPlayingSound) {
            mutePlayer(true);
            trackVolumeButtonClick(true);
        } else {
            mutePlayer(false);
            trackVolumeButtonClick(false);

            if (volume === 0) {
                changeVolume(volumeStepAmount);
            }
        }
    }

    getVolumeIcon() {
        const { t, isPlayingSound } = this.props;

        const unmuteButton = (
            <span className="unmute-button">
                <span
                    className="player-tip"
                    data-tip={t('Unmute')}
                />
                <svg className="player-icon-volumemute">
                    <use xlinkHref="#icon_volumemute" />
                </svg>
            </span>
        );

        const muteButton = (
            <span className="mute-button">
                <span
                    className="player-tip"
                    data-tip={t('Mute')}
                />
                <svg className="player-icon-volumefull">
                    <use xlinkHref="#icon_volumefull" />
                </svg>
            </span>
        );

        return isPlayingSound ? muteButton : unmuteButton;
    }
}

VolumeSliderComponent.propTypes = propTypes;

export const VolumeSlider = translate()(connect(mapStateToProps, mapDispatchToProps)(VolumeSliderComponent));
