import { PlayerUIControls } from './controls';
import { PlayerUIInfo } from './info';
import { BottomControls } from './containers/bottom-controls';
import { StreamStatus } from './containers/top-bar/stream-status';
import { StreamInfo } from './containers/top-bar/stream-info';
import { CollectionTopBar } from 'ui/containers/collections/collection-topbar';
import { MutedSegmentsAlert } from './containers/muted-segments-alert';
import { ExtensionsContainer } from './containers/extensions-container';
import { ExtensionsConfirmation } from './containers/extensions-confirmation';
import { initializeCaptionsSettings } from '../actions/captions';
import { initializePlaybackSettings } from '../actions/playback';
import { initializeSettings } from '../actions/settings';
import * as TwitchEvents from '../backend/events/twitch-event';
import React from 'react';
import ReactDOM from 'react-dom';
import { Provider } from 'react-redux';
import { I18nextProvider } from 'react-i18next';
import { I18N_INSTANCE } from '../lang/i18n';
import { FollowPanel } from './containers/follow-panel';
import { DeadLTVOverlay } from './containers/dead-ltv-overlay';
import { TopBar } from 'ui/containers/top-bar';
import { CollectionSidebar } from './containers/collections/collection-sidebar';
import { CollectionEmptyOverlay } from './containers/collections/collection-empty-overlay';
import { ErrorOverlay } from './containers/overlays/error-overlay';
import { OfflineRecommendationsOverlay } from './containers/recommendations/offline-recommendations';
import { VodPostplayRecommendations } from './containers/recommendations/post-vod-recommendations';
import { Stats } from './containers/stats/stats';
import { LoadingOverlay } from './containers/overlays/loading-overlay';
import { BackgroundBanner } from './containers/background-banner';
import { AdOverlay } from './containers/overlays/ad-overlay';
import { ClosedCaptionsPanel } from './containers/closed-captions/closed-captions-panel';
import { ClosedCaptions } from './containers/closed-captions/closed-captions';
import { PlayButtonOverlay } from './containers/overlays/play-button-overlay';
import { emitWheelEvent } from '../actions/embed-event-emitter';
import { ChromecastOverlay } from './containers/overlays/chromecast-overlay';
import { MatureOverlay } from './containers/overlays/mature-overlay';
import { AgeGateOverlay } from './containers/overlays/age-gate-overlay';
import { SubscribeOverlay } from './containers/overlays/subscribe-overlay';
import { StormWarningOverlay } from './containers/overlays/storm-warning';
import { PreventRefreshDialog } from 'ui/prevent-refresh-dialog';
import { PlayerRoot } from 'ui/containers/player-root';
import { ShowOnHover } from 'ui/containers/common/show-on-hover';
import { UnmuteOverlay } from './containers/overlays/unmute-overlay';
import { QualityRestrictedOverlay } from 'ui/containers/overlays/quality-restricted-overlay';

export function PlayerUI(player, root, store, clipGenerator, opts) {
    let uiInfo;
    let uiControls;
    let uiPreventRefreshDialog;

    // eslint-disable-next-line max-statements
    function init() {
        const options = opts;

        // Root DOM element.
        root.classList.add('player', 'theme--dark');

        // Append the HTML to the DOM.
        root.insertAdjacentHTML('beforeend', require('html-loader!../../html/player.html'));

        const enableCaptionsIfMuted = store.getState().playback.muted;
        store.dispatch(initializeCaptionsSettings(enableCaptionsIfMuted));
        store.dispatch(initializePlaybackSettings());
        store.dispatch(initializeSettings());

        _renderExtensionsConfirmation(store);
        _renderExtensions(store);
        _renderFollowPanel(store);
        _renderCollectionSidebar(store);
        _renderEmptyCollectionOverlay(store);
        _renderDeadLTVOverlay(store);
        _renderStats(store);
        _renderPostVODRecommendationsOverlays(store);
        _renderOfflineRecommendationsOverlays(store);
        _renderLoadingOverlay(store);
        _renderChromeCastOverlay(store);
        _renderMatureOverlay(store);
        _renderAgeGateOverlay(store);
        _renderSubscribeOverlay(store);
        _renderStormWarningOverlay(store);

        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <PlayerRoot root={root}>
                        <UnmuteOverlay />
                        <AdOverlay />
                        <QualityRestrictedOverlay />
                        <ShowOnHover>
                            <TopBar>
                                <CollectionTopBar />
                                <StreamInfo />
                                <StreamStatus />
                            </TopBar>
                            <BottomControls getCurrentTime={player.getCurrentTime} />
                        </ShowOnHover>
                        <MutedSegmentsAlert />
                    </PlayerRoot>
                </I18nextProvider>
            </Provider>,
            document.getElementById('root-player')
        );

        ReactDOM.render(
            <Provider store={store}>
                <ClosedCaptions />
            </Provider>,
            document.getElementById('js-player-captions-container')
        );

        ReactDOM.render(
            <Provider store={store}>
                <BackgroundBanner />
            </Provider>,
            document.getElementById('js-background-banner')
        );

        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <PlayButtonOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-paused-overlay')
        );

        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <ErrorOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('error-overlay')
        );

        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <ClosedCaptionsPanel />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-cc-panel')
        );

        // Now we can initialize the subcomponents.
        clipGenerator.hookUpForm(root);
        uiInfo = new PlayerUIInfo(player, root, store, options);
        uiControls = new PlayerUIControls(root, store, options);
        uiPreventRefreshDialog = new PreventRefreshDialog(player, root, store, options);
        root.setAttribute('data-initializing', true);

        player.addEventListener(TwitchEvents.PLAYER_INIT, function() {
            root.setAttribute('data-initializing', false);
        });

        // Add wheel listener to propogate wheel events to embedded container
        root.addEventListener('wheel', function(e) {
            store.dispatch(emitWheelEvent(e));
        });
    }

    function _renderLoadingOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <LoadingOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-loading-overlay')
        );
    }

    function _renderStats(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <Stats />
                </I18nextProvider>
            </Provider>,
            document.getElementById('stats')
        );
    }

    function _renderCollectionSidebar(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <CollectionSidebar />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-collection-sidebar')
        );
    }

    function _renderEmptyCollectionOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <CollectionEmptyOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('empty-collection-overlay')
        );
    }

    // eslint-disable-next-line no-unused-vars
    function _renderFollowPanel(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <FollowPanel />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-follow-panel')
        );
    }

    // eslint-disable-next-line no-unused-vars
    function _renderDeadLTVOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <DeadLTVOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-dead-ltv-overlay')
        );
    }

    function _renderExtensionsConfirmation(store) {
        const extensionsConfirmationHost = document.getElementById('js-player-extension-confirmation-root');
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <ExtensionsConfirmation store={store} />
                </I18nextProvider>
            </Provider>,
            extensionsConfirmationHost
        );
    }

    function _renderExtensions(store) {
        const extensionsHost = document.getElementById('js-player-extension-root');
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <ExtensionsContainer store={store} />
                </I18nextProvider>
            </Provider>,
            extensionsHost
        );
    }

    function _renderOfflineRecommendationsOverlays(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <OfflineRecommendationsOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-player-offline-recommendations-overlay')
        );
    }

    function _renderPostVODRecommendationsOverlays(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <VodPostplayRecommendations />
                </I18nextProvider>
            </Provider>,
            document.getElementById('js-player-post-vod-recommendations-overlay')
        );
    }

    function _renderChromeCastOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <ChromecastOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('player-chromecast')
        );
    }

    function _renderMatureOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <MatureOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('player-mature')
        );
    }

    function _renderAgeGateOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <AgeGateOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('player-age-gate-overlay')
        );
    }

    function _renderSubscribeOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <SubscribeOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('player-subscribe-overlay')
        );
    }

    function _renderStormWarningOverlay(store) {
        ReactDOM.render(
            <Provider store={store}>
                <I18nextProvider i18n={I18N_INSTANCE}>
                    <StormWarningOverlay />
                </I18nextProvider>
            </Provider>,
            document.getElementById('storm-warning-overlay')
        );
    }

    // eslint-disable-next-line max-statements
    this.destroy = function() {
        uiInfo.destroy();
        uiControls.destroy();
        uiPreventRefreshDialog.destroy();
    };

    init();
}
