function buildClipsEndpoint(path) {
    return `https://clips.twitch.tv/api/v2/${path}`;
}

export function incrementClipViewCount(clipSlug) {
    fetch(buildClipsEndpoint(`clips/${clipSlug}/view`));
}

export function fetchRecommendationsSets(clipSlug, sets) {
    return fetch(buildClipsEndpoint(`clips/${clipSlug}/related-clips-sets?sets=${sets.join(',')}`), {
        method: 'GET',
    }).then(response => {
        return response.json();
    }).then(parsedResponse => {
        if (parsedResponse.status === 'error') {
            return Promise.reject({
                apiError: true,
                status: parsedResponse.status,
                message: parsedResponse.message,
            });
        }
        return {
            apiError: false,
            clipSets: normalizedClipSets(parsedResponse),
        };
    }).catch(errorResponse => {
        if (!errorResponse.response) {
            return {
                apiError: true,
                message: errorResponse.message,
            };
        }

        const statusCode = parseInt(errorResponse.response.status, 10);

        return Promise.reject({
            apiError: true,
            status: statusCode,
            message: errorResponse.message,
        });
    });
}

function normalizedClipSets(clipSets) {
    return Object.keys(clipSets).reduce((newClipSets, clipsSetKey) => {
        // eslint-disable-next-line no-param-reassign
        newClipSets[clipsSetKey] = clipSets[clipsSetKey].map(clip => {
            return normalizedClipInfoFromRecommendations(clip);
        });

        return newClipSets;
    }, {});
}

/**
 * Normalizes api's clipInfo data from clips recommendations
 *
 * @param {Object} clip recommendations - API's clipInfo data object
 * @return {Object} data in the form of state's streamMetadata property
 */
function normalizedClipInfoFromRecommendations(clipInfo) {
    return {
        id: clipInfo.id,
        channel: {
            channelUrl: clipInfo.broadcaster_channel_url,
            displayName: clipInfo.broadcaster_display_name,
            name: clipInfo.broadcaster_login,
            id: clipInfo.broadcaster_id,
            logo: clipInfo.broadcaster_logo,
        },
        curator: {
            channelUrl: clipInfo.curator_channel_url,
            displayName: clipInfo.curator_display_name,
            name: clipInfo.curator_login,
            id: clipInfo.curator_id,
            logo: clipInfo.curator_logo,
        },
        game: clipInfo.game,
        url: clipInfo.url,
        title: clipInfo.title,
        thumbnails: clipInfo.thumbnails,
        createdAt: clipInfo.created_at,
        duration: clipInfo.duration,
        embedHtml: clipInfo.embed_html,
        embedUrl: clipInfo.embed_url,
        language: clipInfo.language,
        slug: clipInfo.slug,
        viewCount: clipInfo.views,
        vod: {
            id: clipInfo.vod_id,
            offset: clipInfo.vod_offset,
            url: clipInfo.vod_url,
            previewImageUrl: clipInfo.vod_preview_image_url,
        },
        segments: clipInfo.segments,
        communities: clipInfo.communities,
        editUrl: clipInfo.edit_url,
        infoUrl: clipInfo.info_url,
        statusUrl: clipInfo.status_url,
        previewImage: clipInfo.preview_image,
    };
}
