import React from 'react';
import PropTypes from 'prop-types';
import { LoadingOverlay } from 'ui/containers/overlays/loading-overlay';
import { BottomControls } from 'ui/containers/bottom-controls';
import { PLAYER_CLIPS_EMBED } from 'util/player-type';
import { PlayerUIControls } from 'ui/controls';
import { PlayerUIInfo } from 'ui/info';
import { PlayerRoot } from 'ui/containers/player-root';
import { ShowOnHover } from 'ui/containers/common/show-on-hover';
import { ClipInfoOverlay } from 'ui/player-types/clips/overlays/clip-info-overlay/clip-info-overlay';
import { ClipsLoadingTimeSpentTracker } from 'ui/player-types/clips/utils/tracking/clips-loading-time-spent-tracker';
import { ClipsMinutesWatched } from 'ui/player-types/clips/utils/tracking/clips-minutes-watched';
import { PlayerClickHandlerOverlay } from 'ui/containers/overlays/player-click-handler-overlay';
import { ClipsBenchmarkTracking } from 'ui/player-types/clips/utils/tracking/clips-benchmark-tracking';
import { ClipPlayButtonOverlay } from 'ui/player-types/clips/overlays/clip-play-button-overlay-container';
import {
    ClipKeyboardShortcutsOverlayContainer,
} from 'ui/player-types/clips/overlays/clip-keyboard-shortcuts-overlay/clip-keyboard-shortcuts-overlay-container';
import {
    ClipShareOverlayContainer,
} from 'ui/player-types/clips/overlays/clip-share-overlay/clip-share-overlay-container';
import {
    ClipPostplayRecommendations,
} from 'ui/player-types/clips/overlays/clip-postplay-recommendations/clip-postplay-recommendations';
import { CLIPS_GOT_EM } from 'experiments';
require ('./styles/clips-player.sass');

const contextTypes = {
    store: PropTypes.object.isRequired,
};

const propTypes = {
    player: PropTypes.object.isRequired,
    root: PropTypes.object.isRequired,
};

export default class ClipsEmbedPlayer extends React.Component {
    constructor() {
        super(...arguments);
    }

    componentDidMount() {
        const { player, root } = this.props;
        const store = this.context.store;
        const options = {
            player: PLAYER_CLIPS_EMBED,
        };

        // This experiment is used for cross app latency tracking purposes
        // The experiment value is never used in app, but this must remain
        store.getState().experiments.get(CLIPS_GOT_EM);

        // Root DOM element.
        root.classList.add('player', 'theme--dark');
        root.insertAdjacentHTML('beforeend', require('html-loader!./html/clips.html'));

        const uiInfo = new PlayerUIInfo(player, root, store, options);
        const uiControls = new PlayerUIControls(root, store, options);
        const loadingTimeSpentTracker = new ClipsLoadingTimeSpentTracker(store, player);
        const minWatchTracker = new ClipsMinutesWatched(store);
        const benchmarkTracking = new ClipsBenchmarkTracking(store);

        this.setState({
            legacyUI: [
                uiInfo,
                uiControls,
                loadingTimeSpentTracker,
                minWatchTracker,
                benchmarkTracking,
            ],
        });

        benchmarkTracking.trackAppReady();
    }

    componentWillUnmount() {
        this.state.legacyUI.forEach(uiController => {
            uiController.destroy();
        });
    }

    render() {
        return (
            <div
                id="clips_embed"
            >
                {/* Players are assumed to have double click behavior so it breaks without the below */}
                <div className="js-control-fullscreen-overlay" />
                <ClipPlayButtonOverlay />
                <PlayerClickHandlerOverlay />
                <ClipShareOverlayContainer />
                <ClipKeyboardShortcutsOverlayContainer />
                <LoadingOverlay />
                <PlayerRoot root={this.props.root}>
                    <ShowOnHover>
                        <ClipInfoOverlay />
                        <ClipPostplayRecommendations />
                        <BottomControls getCurrentTime={this.props.player.getCurrentTime} />
                    </ShowOnHover>
                </PlayerRoot>
            </div>
        );
    }
}

ClipsEmbedPlayer.contextTypes = contextTypes;
ClipsEmbedPlayer.propTypes = propTypes;
