import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { translate } from 'react-i18next';
import { NumberFormatter } from 'ui/player-types/clips/utils/number-formatter';
import { ttAnalyticsQuery, Links } from 'ui/player-types/clips/utils/link-helper';
import { ImageFallback } from 'ui/player-types/clips/common/image-fallback';
import {
    TRACKING_MEDIUM_EMBED,
    TRACKING_CONTENT_TITLE,
    TRACKING_CONTENT_CHANNEL,
    TRACKING_CONTENT_PROFILE_IMAGE,
    TRACKING_CONTENT_GAME,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import {
    BroadcasterPlayingGameLink,
} from 'ui/player-types/clips/overlays/clip-info-overlay/broadcaster-playing-game-link';
import { DEFAULT_AVATAR_URL } from 'settings';
import bowser from 'bowser';

function channelURL(broadcasterLogin, isForImage) {
    const channelURL = Links.default.getClipsOnChannelPageLink(broadcasterLogin);

    if (bowser.android) {
        return Links.android.getChannelLink(broadcasterLogin, channelURL);
    }

    return ttAnalyticsQuery(
        channelURL,
        TRACKING_MEDIUM_EMBED,
        isForImage ? TRACKING_CONTENT_PROFILE_IMAGE : TRACKING_CONTENT_CHANNEL
    );
}

function channelURLName(broadcasterLogin) {
    return channelURL(broadcasterLogin, false);
}

function channelURLImage(broadcasterLogin) {
    return channelURL(broadcasterLogin, true);
}

function gameUrl(game) {
    if (game) {
        const gameUrl = Links.default.getClipsOnGamePageLink(game);
        return ttAnalyticsQuery(gameUrl, TRACKING_MEDIUM_EMBED, TRACKING_CONTENT_GAME);
    }

    return '';
}

const propTypes = {
    logoUrl: PropTypes.string.isRequired,
    title: PropTypes.string,
    clipURL: PropTypes.string.isRequired,
    broadcasterDisplayName: PropTypes.string.isRequired,
    broadcasterLogin: PropTypes.string.isRequired,
    viewCount: PropTypes.number,
    game: PropTypes.string,
    gameLink: PropTypes.string,
    t: PropTypes.func.isRequired,
};

const mapStateToProps = ({ streamMetadata }) => ({
    title: streamMetadata.title,
    clipURL: streamMetadata.url,
    game: streamMetadata.game,
    broadcasterDisplayName: streamMetadata.channel.displayName,
    broadcasterLogin: streamMetadata.channel.name,
    logoUrl: streamMetadata.channel.logo,
    isLive: true, // TODO
    viewCount: streamMetadata.viewCount,
});

export function ClipInfoOverlayComponent({
    logoUrl,
    title,
    clipURL,
    broadcasterDisplayName,
    broadcasterLogin,
    game,
    viewCount,
    t,
}) {
    if (!broadcasterLogin) { // when metadata is not loaded, return null
        return null;
    }

    return (
        <div className={'pl-controls-top clip-player-userinfo'}>
            <a
                href={channelURLImage(broadcasterLogin)}
                target="_blank"
                className="clip-userinfo__picture"
            >
                <ImageFallback src={logoUrl} altSrc={DEFAULT_AVATAR_URL} alt={t('profile image')} />
            </a>
            <div className="clip-userinfo__meta-container">
                <div className="clip-userinfo__name ellipsis">
                    { title &&
                        <a
                            href={ttAnalyticsQuery(clipURL, TRACKING_MEDIUM_EMBED, TRACKING_CONTENT_TITLE)}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="clip-userinfo__title-link"
                        >
                            {title}
                        </a>
                    }
                </div>
                <div className="clip-userinfo__title">
                    <BroadcasterPlayingGameLink
                        broadcasterDisplayName={broadcasterDisplayName}
                        channelURL={channelURLName(broadcasterLogin)}
                        game={game}
                        gameUrl={gameUrl(game)}
                    />
                </div>
                <div className="clip-userinfo__title">
                    {
                        t('{{numberOfViews}} view', {
                            numberOfViews: new NumberFormatter(viewCount).inCommas(),
                            count: viewCount,
                        })
                    }
                </div>
            </div>
        </div>
    );
}

ClipInfoOverlayComponent.propTypes = propTypes;

export const ClipInfoOverlay = translate()(
    connect(mapStateToProps)(ClipInfoOverlayComponent)
);
