import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { ImageFallback } from 'ui/player-types/clips/common/image-fallback';
import { DEFAULT_AVATAR_URL } from 'settings';

import { ClipTinySVG } from 'ui/player-types/clips/common/svg/clip-tiny';
import { VodTinySVG } from 'ui/player-types/clips/common/svg/vod-tiny';
import {
    ITEM_TYPE_CLIP,
    ITEM_TYPE_LIVE,
    ITEM_TYPE_VOD,
} from 'ui/player-types/clips/utils/embed-postplay-overlay-selectors';
import { NumberFormatter } from 'ui/player-types/clips/utils/number-formatter';
import { translate } from 'react-i18next';
import classNames from 'classnames';

const CARD_BASE_CLASSES = [
    'pl-pill',
    'pl-pill--black',
    'pl-pd-05',
    'pl-flex--inline',
    'pl-flex--verticalCenter',
    'pl-flex__item--noGrow',
];

const LIVE_CLASSES = classNames(
    ...CARD_BASE_CLASSES,
    'pl-pill--red'
);

const CLIP_VOD_CLASSES = classNames(
    ...CARD_BASE_CLASSES,
    'pl-pill--clips-overlay'
);

const propTypes = {
    linkUrl: PropTypes.string.isRequired,
    itemType: PropTypes.string.isRequired,
    imageUrl: PropTypes.string.isRequired,
    title: PropTypes.string.isRequired,
    viewCount: PropTypes.number,
    game: PropTypes.string.isRequired,
    displayName: PropTypes.string.isRequired,
    trackSelectionClick: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
    cardDimensions: PropTypes.shape({
        thumb: PropTypes.shape({
            width: PropTypes.number.isRequired,
            height: PropTypes.number.isRequired,
        }).isRequired,
        card: PropTypes.shape({
            width: PropTypes.number.isRequired,
            height: PropTypes.number.isRequired,
        }).isRequired,
    }).isRequired,
};

export class CardContentPreviewComponent extends Component {
    clipSubtext() {
        const { viewCount, game, t } = this.props;

        const translatedViews = t('{{numberOfViews}} view', {
            numberOfViews: new NumberFormatter(parseInt(viewCount, 10)).inCommas(),
            count: parseInt(viewCount, 10),
        });

        if (game) {
            return `${translatedViews} • ${game}`;
        }

        return translatedViews;
    }

    getContentSubtext() {
        const { itemType, game, displayName, t } = this.props;

        if (itemType === ITEM_TYPE_CLIP) {
            return this.clipSubtext();
        } else if (game) {
            return t('{{displayName}} playing {{game}}', {
                displayName,
                game,
            });
        // Below is fallback for VODs and LIVE without game data
        } else if (itemType === ITEM_TYPE_LIVE) {
            return t('{{displayName}} streaming on Twitch', { displayName });
        } else if (itemType === ITEM_TYPE_VOD) {
            // eslint-disable-next-line quotes
            return t(`{{displayName}}'s video on Twitch`, { displayName });
        }

        return null;
    }

    render() {
        const { cardDimensions, linkUrl, imageUrl, title, itemType, displayName, trackSelectionClick, t } = this.props;

        return (
            <div className="pl-flex__item theme--dark cem-rec-two-up">
                <div className="pl-card cem-rec-card pos-relative">
                    <a
                        href={linkUrl}
                        onMouseDown={trackSelectionClick}
                        className="pl-card__layout cem-rec__layout"
                        style={{
                            width: cardDimensions.card.width,
                            height: cardDimensions.card.height,
                        }}
                        target="_blank"
                    >
                        <div className="cem-rec-card__type strong">
                            {itemType === ITEM_TYPE_LIVE &&
                                <div className={LIVE_CLASSES}>
                                    <div className="cl-live-indicator inline-block relative" />
                                    <div>{t('Live')}</div>
                                </div>
                            }
                            {itemType === ITEM_TYPE_VOD &&
                                <div className={CLIP_VOD_CLASSES}>
                                    <figure className="inline-block cem-rec__icon cem-rec__icon--overlay">
                                        <VodTinySVG />
                                    </figure>
                                    <div>{t('Full Video')}</div>
                                </div>
                            }
                            {itemType === ITEM_TYPE_CLIP &&
                                <div className={CLIP_VOD_CLASSES}>
                                    <figure className="inline-block cem-rec__icon cem-rec__icon--overlay">
                                        <ClipTinySVG />
                                    </figure>
                                    <div>{t('Clip')}</div>
                                </div>
                            }
                        </div>

                        <figure
                            className="pl-card__img cem-rec__img"
                            style={{
                                width: cardDimensions.thumb.width,
                                height: cardDimensions.thumb.height,
                            }}
                        >
                            <ImageFallback src={imageUrl} altSrc={DEFAULT_AVATAR_URL} alt={t('profile image')} />
                        </figure>

                        <div className="pl-card__body">
                            <div className="pl-card__title card__title--cem ellipsis">
                                {itemType === ITEM_TYPE_LIVE &&
                                    <span className="pl-strong">
                                        {t('Watch {{broadcaster}} Live', { broadcaster: displayName })}
                                    </span>
                                }
                                {itemType === ITEM_TYPE_VOD &&
                                    <span className="pl-strong">{t('Continue Watching')}</span>
                                }
                                {itemType === ITEM_TYPE_CLIP &&
                                    <span>{title}</span>
                                }
                            </div>
                            <div className="pl-card__info card__info--cem ellipsis">
                                {this.getContentSubtext()}
                            </div>
                        </div>
                    </a>
                </div>
            </div>
        );
    }
}

CardContentPreviewComponent.propTypes = propTypes;

export const CardContentPreview = translate()(CardContentPreviewComponent);
