import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import {
    PostplayContentTwoLayout,
} from 'ui/player-types/clips/overlays/clip-postplay-recommendations/postplay-content-two-layout';
import { DEFAULT_AVATAR_URL } from 'settings';
import assign from 'lodash/assign';
import { fetchClipRecommendations } from 'actions/clip-recommendations';
import { postPlayContentDataSelector } from 'ui/player-types/clips/utils/embed-postplay-overlay-selectors';
import { preloadImages } from 'ui/player-types/clips/utils/image-helper';
import {
    TRACKING_MEDIUM_EMBED,
    TRACKING_ITEM_SECTION_CLICK,
    TRACKING_ITEM_SECTION_LOAD,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import { clipsRecommendationProperties } from 'ui/player-types/clips/utils/tracking/base-properties';
import classNames from 'classnames';
import { fetchStreamInfo, fetchChannelInfo, NOT_AVAILABLE } from 'actions/broadcaster-info';
import { broadcasterLiveStatus } from 'state/broadcaster-info';

export const MAX_CONTENT_TO_SHOW = 2;

const WRAPPER_CLASSES = classNames(
    'pl-flex',
    'pl-flex--verticalCenter',
    'pl-flex--horizontalCenter',
    'player-recommend',
    'pl-flex--nowrap',
    'pl-rec-wrapper'
);

function formatSelectionLoadData(contentData) {
    return {
        /* eslint-disable camelcase */
        rendered_item_list: joinedData(contentData, 'itemId'),
        rendered_item_count: contentData.length,
        rendered_items_viewcounts: joinedData(contentData, 'viewCount'),
        num_recs_visible: contentData.length,
        rendered_items_types: joinedData(contentData, 'itemType'),
        section_index: 0,
        /* eslint-enable camelcase */
    };
}

function joinedData(contentData, propertyName) {
    return contentData.map(
        data => data[propertyName]
    ).map(
        trackingData => trackingData === null ? 'null' : trackingData
    ).join(',');
}

const propTypes = {
    analyticsTracker: PropTypes.object.isRequired,
    recommendationTrackingProperties: PropTypes.object.isRequired,
    showOverlay: PropTypes.bool.isRequired,
    channelName: PropTypes.string.isRequired,
    postPlayContentData: PropTypes.arrayOf(PropTypes.shape({
        itemType: PropTypes.string.isRequired,
        linkUrl: PropTypes.string.isRequired,
        title: PropTypes.string.isRequired,
        itemId: PropTypes.string.isRequired,
        viewCount: PropTypes.number,
        channel: PropTypes.string.isRequired,
        displayName: PropTypes.string.isRequired,
        game: PropTypes.string.isRequired,
    })),
    fetchRecommendationsSets: PropTypes.func.isRequired,
    hasFetchedRelatedClips: PropTypes.bool.isRequired,
    hasFetchedBroadcasterInfo: PropTypes.bool.isRequired,
    clipSlug: PropTypes.string,
    playerDimensions: PropTypes.shape({
        height: PropTypes.number,
        width: PropTypes.number,
    }).isRequired,
};

const mapStateToProps = state => {
    const { streamMetadata, playback, playerDimensions, analyticsTracker } = state;
    const hasFetchedRelatedClips =
        Boolean(state.clipRecommendations.hasFetched[streamMetadata.slug]);
    const canShowPostplayOverlay = playback.ended;

    return {
        clipSlug: streamMetadata.slug,
        channelName: streamMetadata.channel.name,
        showOverlay: hasFetchedRelatedClips && canShowPostplayOverlay,
        postPlayContentData: postPlayContentDataSelector(state, MAX_CONTENT_TO_SHOW),
        hasFetchedRelatedClips,
        playerDimensions,
        hasFetchedBroadcasterInfo: broadcasterLiveStatus(state, streamMetadata.channel.name) !== NOT_AVAILABLE,
        analyticsTracker,
        recommendationTrackingProperties: clipsRecommendationProperties(state),
    };
};

export const mapDispatchToProps = dispatch => ({
    fetchRecommendationsSets(channelName) {
        dispatch(fetchClipRecommendations());
        // TODO: move these to init and deal with tracking firing before these return | CLIP-1881
        dispatch(fetchChannelInfo(channelName));
        dispatch(fetchStreamInfo(channelName));
    },
});

export class ClipPostplayRecommendationsComponent extends Component {
    constructor() {
        super();

        this.state = {
            isImagePreloadingComplete: false,
            postPlayContentData: [],
        };

        this.trackedForVideoEnd = false;
        this.attemptedRecommendationsFetched = false;
    }

    handleTrackForClickGenerator({ itemId, viewCount, channel, game, itemType }, index) {
        return () => {
            this.props.analyticsTracker.clipsTrackEvent(
                TRACKING_ITEM_SECTION_CLICK,
                assign(
                    {},
                    this.props.recommendationTrackingProperties,
                    {
                        /* eslint-disable camelcase */
                        item_id: itemId,
                        item_viewcount: viewCount,
                        channel,
                        game,
                        item_type: itemType,
                        item_index: index,
                        location: TRACKING_MEDIUM_EMBED,
                        section_index: 0,
                        /* eslint-enable camelcase */
                    }
                )
            );
        };
    }

    // eslint-disable-next-line complexity
    componentWillReceiveProps(nextProps) {
        const {
            hasFetchedRelatedClips,
            hasFetchedBroadcasterInfo,
            clipSlug,
            fetchRecommendationsSets,
            channelName,
        } = nextProps;

        if (!this.attemptedRecommendationsFetched && clipSlug && channelName) {
            fetchRecommendationsSets(channelName);
            this.attemptedRecommendationsFetched = true;
        }

        if (hasFetchedRelatedClips && clipSlug && hasFetchedBroadcasterInfo && !this.state.isImagePreloadingComplete) {
            this.handleContentPreLoading(nextProps);
        }

        if (nextProps.showOverlay && !this.trackedForVideoEnd) {
            this.props.analyticsTracker.clipsTrackEvent(
                TRACKING_ITEM_SECTION_LOAD,
                assign(
                    {},
                    this.props.recommendationTrackingProperties,
                    formatSelectionLoadData(this.state.postPlayContentData)
                )
            );

            this.trackedForVideoEnd = true;
        } else if (this.trackedForVideoEnd && !nextProps.showOverlay) {
            this.trackedForVideoEnd = false;
        }
    }

    handleContentPreLoading({ postPlayContentData }) {
        const imagesToPreload = postPlayContentData.map(contentComp => contentComp.imageUrl);
        preloadImages([...imagesToPreload, DEFAULT_AVATAR_URL]);

        const twoPostPlay = postPlayContentData.slice(0, 2);
        const postPlayContentDataWithTracking = twoPostPlay.map(item => assign(
            {},
            item,
            { trackSelectionClick: this.handleTrackForClickGenerator(item) }
        ));

        this.setState({
            isImagePreloadingComplete: true,
            postPlayContentData: postPlayContentDataWithTracking,
        });
    }

    render() {
        if (!this.props.showOverlay) {
            return null;
        }

        return (
            <div className={WRAPPER_CLASSES}>
                <PostplayContentTwoLayout
                    postPlayContentData={this.state.postPlayContentData}
                    playerWidth={this.props.playerDimensions.width}
                    playerHeight={this.props.playerDimensions.height}
                />
            </div>
        );
    }
}

ClipPostplayRecommendationsComponent.propTypes = propTypes;

export const ClipPostplayRecommendations =
    connect(mapStateToProps, mapDispatchToProps)(ClipPostplayRecommendationsComponent);
