import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { CardContentPreview } from 'ui/player-types/clips/overlays/clip-postplay-recommendations/card-content-preview';

const ASPECT_RATIO = 16 / 9;

const CARD_HORIZONTAL_PADDING = 15;
const NUM_PAD_EDGES = 3; // two cards horizontally -> 3 edges to pad.
const TOTAL_WIDTH_PADDING = CARD_HORIZONTAL_PADDING * NUM_PAD_EDGES;
const DEFAULT_PADDED_WIDTH = 1280;

const VERTICAL_PADDING = 230;
const VERTICAL_PADDING_SM = 130;
const PAD_HEIGHT_THREHOLD = 350; // Determines which vertical padding is used
const CARD_BODY_HEIGHT = 45; // card title + info

const WRAPPER_CLASSES = classNames(
    'pl-flex',
    'pl-flex--verticalCenter',
    'pl-flex--horizontalCenter',
    'pl-flex--nowrap',
    'cem-rec-two-up-wrap',
    'pl-mg-x-1'
);

function getCardDimensions(playerHeight, playerWidth) {
    const paddedW = (playerWidth > DEFAULT_PADDED_WIDTH) ? DEFAULT_PADDED_WIDTH : playerWidth - TOTAL_WIDTH_PADDING;
    const vPad = (playerHeight < PAD_HEIGHT_THREHOLD) ? VERTICAL_PADDING_SM : VERTICAL_PADDING;
    const paddedH = playerHeight - vPad;

    let thumbW = paddedW / 2;
    let thumbH = thumbW * (1 / ASPECT_RATIO);

    if (thumbH > paddedH) {
        thumbW = paddedH * ASPECT_RATIO;
        thumbH = paddedH;
    }

    return {
        thumb: {
            width: thumbW,
            height: thumbH,
        },
        card: {
            width: thumbW,
            height: thumbH + CARD_BODY_HEIGHT,
        },
    };
}

const propTypes = {
    postPlayContentData: PropTypes.arrayOf(PropTypes.object).isRequired,
    playerWidth: PropTypes.number.isRequired,
    playerHeight: PropTypes.number.isRequired,
};

export function PostplayContentTwoLayout({ postPlayContentData, playerWidth, playerHeight }) {
    return (
        <div className={WRAPPER_CLASSES}>
            {postPlayContentData.map(contentData => {
                return (
                    <CardContentPreview
                        linkUrl={contentData.linkUrl}
                        itemType={contentData.itemType}
                        imageUrl={contentData.imageUrl}
                        title={contentData.title}
                        viewCount={contentData.viewCount}
                        game={contentData.game}
                        displayName={contentData.displayName}
                        trackSelectionClick={contentData.trackSelectionClick}
                        cardDimensions={getCardDimensions(playerHeight, playerWidth)}
                        key={contentData.linkUrl}
                    />
                );
            }) }
        </div>
    );
}

PostplayContentTwoLayout.propTypes = propTypes;
