import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { translate } from 'react-i18next';
import classNames from 'classnames';
import { ShareIcon } from 'ui/player-types/clips/overlays/clip-share-overlay/share-icon';
import { CrossSVG } from 'ui/player-types/clips/common/svg/cross';
import { EmbedSVG } from 'ui/player-types/clips/common/svg/embed';
import { SelectableInput } from 'ui/player-types/clips/common/selectable-input';
import { ChevronLeftSVG } from 'ui/player-types/clips/common/svg/chevron-left';
import { FacebookSVG } from 'ui/player-types/clips/common/svg/facebook';
import { RedditSVG } from 'ui/player-types/clips/common/svg/reddit';
import { TwitterSVG } from 'ui/player-types/clips/common/svg/twitter';
import {
    TRACKING_VIDEO_SHARE_EVENT,
    TRACKING_MEDIUM_EMBED,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import CopyToClipboard from 'react-copy-to-clipboard';

const EMBED_WRAPPER_CLASSES = classNames(
    'flex',
    'flex--verticalCenter',
    'flex--horizontalCenter',
    'flex--nowrap',
    'cs-wrapper',
    'player-overlay--recommendations'
);

const EMBED_SELECTABLE_CLASSES = classNames(
    'cs-input',
    'pl-flex',
    'pl-flex--horizontalCenter',
    'pl-flex--verticalCenter',
    'pl-flex--nowrap'
);

const SHARE_WRAPPER_CLASSES = classNames(
    'pl-flex',
    'pl-flex--verticalCenter',
    'pl-flex--horizontalCenter',
    'cs-wrapper',
    'player-overlay--recommendations'
);

const SHARE_ICON_WRAPPER_CLASSES = classNames(
    'cs-shareIcons',
    'pl-flex',
    'pl-flex--horizontalCenter',
    'pl-flex--verticalCenter',
    'pl-flex--nowrap'
);

export class ClipShareOverlayComponent extends Component {
    constructor() {
        super();

        this.hideEmbedLink = this.setShowEmbedLink.bind(this, false);
        this.showEmbedLink = this.setShowEmbedLink.bind(this, true);

        this.trackLinkClick = this.trackLinkClick.bind(this);
        this.trackShareClick = this.trackShareClick.bind(this);

        this.state = { showEmbedLink: false };
    }

    trackLinkClick() {
        this.trackShareClick('link');
    }

    trackShareClick(sharePlatform) {
        this.props.analyticsTracker.clipsTrackEvent(TRACKING_VIDEO_SHARE_EVENT, {
            // eslint-disable-next-line camelcase
            share_platform: sharePlatform,
            location: TRACKING_MEDIUM_EMBED,
        });
    }

    setShowEmbedLink(showEmbedLink) {
        this.setState({ showEmbedLink });
    }

    render() {
        const { clipUrl, clipEmbedUrl, hideSharing, t } = this.props;
        const { showEmbedLink } = this.state;

        const closeButton = (
            <button className="pl-button pl-button--text cs-close-button" onClick={hideSharing}>
                <figure className="cs-nav-icon">
                    <CrossSVG />
                </figure>
            </button>
        );

        if (showEmbedLink) {
            return (
                <div className={EMBED_WRAPPER_CLASSES}>
                    <button className="pl-button pl-button--text cs-back-button" onClick={this.hideEmbedLink}>
                        <figure className="cs-nav-icon">
                            <ChevronLeftSVG />
                        </figure>
                    </button>
                    { closeButton }
                    <div className="flex flex--verticalCenter flex--horizontalCenter cs-layout theme--dark">
                        <div className="cs-title pl-align-center">
                            <h1 className="pl-pd-b-0">{t('Embed')}</h1>
                        </div>
                        <div className="cs-divider" />
                        <div className={EMBED_SELECTABLE_CLASSES}>
                            <SelectableInput
                                id="embed"
                                label={t('Embed Code')}
                                value={clipEmbedUrl}
                                onSelect={this.trackLinkClick}
                            />
                            <CopyToClipboard
                                text={clipEmbedUrl}
                            >
                                <button
                                    target="_blank"
                                    className="pl-flex__item cs-button pl-button"
                                    onClick={this.trackLinkClick}
                                >
                                    {t('Copy')}
                                </button>
                            </CopyToClipboard>
                        </div>
                    </div>
                </div>
            );
        }

        return (
            <div className={SHARE_WRAPPER_CLASSES}>
                { closeButton }
                <div className="pl-flex pl-flex--verticalCenter pl-flex--horizontalCenter cs-layout theme--dark">
                    <div className="cs-title pl-align-center">
                        <h1 className="pl-mg-b-2">{t('Sharing is Caring')}</h1>
                    </div>
                    <div className={SHARE_ICON_WRAPPER_CLASSES}>
                        <ShareIcon
                            companySVG={<TwitterSVG />}
                            companyName="twitter"
                            onClick={this.trackShareClick}
                        />
                        <ShareIcon
                            companySVG={<FacebookSVG />}
                            companyName="facebook"
                            onClick={this.trackShareClick}
                        />
                        <ShareIcon
                            companySVG={<RedditSVG />}
                            companyName="reddit"
                            onClick={this.trackShareClick}
                        />
                        <button className="cs-icon cs-icon--embed" onClick={this.showEmbedLink}>
                            <EmbedSVG />
                        </button>
                    </div>
                    <div className="cs-divider" />
                    <div className="cs-input pl-flex pl-flex--horizontalCenter pl-flex--verticalCenter pl-flex--nowrap">
                        <SelectableInput
                            id="link"
                            label={t('Link')}
                            value={clipUrl}
                            onSelect={this.trackLinkClick}
                        />
                        <CopyToClipboard
                            text={clipUrl}
                        >
                            <button
                                target="_blank"
                                className="pl-flex__item button cs-button pl-button"
                                onClick={this.trackLinkClick}
                            >
                                {t('Copy')}
                            </button>
                        </CopyToClipboard>
                    </div>
                </div>
            </div>
        );
    }
}

ClipShareOverlayComponent.propTypes = {
    clipUrl: PropTypes.string.isRequired,
    hideSharing: PropTypes.func.isRequired,
    clipEmbedUrl: PropTypes.string.isRequired,
    analyticsTracker: PropTypes.object.isRequired,
    t: PropTypes.func.isRequired,
};

export const ClipShareOverlay = translate()(ClipShareOverlayComponent);
