import {
    TRACKING_PAGEVIEW_EVENT,
    TRACKING_BENCHMARK_FETCH_START,
    TRACKING_BENCHMARK_COMPLETE_TRANSITION,
    TRACKING_BENCHMARK_VIDEO_STARTED,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import { UIStateSubscriber } from 'ui/state-subscriber';
import { NOT_AVAILABLE } from 'actions/broadcaster-info';
import { broadcasterLiveStatus } from 'state/broadcaster-info';
import isFinite from 'lodash/isFinite';

export const MILLIS_IN_SECOND = 1000;

export const performanceFetchStartSelector = ({ window }) => {
    if (window.performance && window.performance.timing && window.performance.timing.fetchStart) {
        return window.performance.timing.fetchStart;
    } else if (window.Benchmark) {
        return window.Benchmark;
    }
    return undefined;
};

export class ClipsBenchmarkTracking extends UIStateSubscriber {
    constructor(store) {
        super();

        this.store = store;
        this.trackPageview = this.trackPageview.bind(this);
        this.handleStateChange = this.handleStateChange.bind(this);
        this.shouldSendEvents = false;
        this.eventsQueue = [];

        this.subscribe(store, ['streamMetadata', 'playback', 'broadcasterInfo'], state => {
            this.handleStateChange(state);
        });

        this.hasTrackedEventMap = {
            [TRACKING_PAGEVIEW_EVENT]: false,
            [TRACKING_BENCHMARK_FETCH_START]: false,
            [TRACKING_BENCHMARK_COMPLETE_TRANSITION]: false,
            [TRACKING_BENCHMARK_VIDEO_STARTED]: false,
        };

        this.trackFetchStart();

        this.handleStateChange(store.getState());
    }

    setEventToTracked(eventName) {
        this.hasTrackedEventMap[eventName] = true;
    }

    hasEventBeenTracked(eventName) {
        return this.hasTrackedEventMap[eventName];
    }

    handleStateChange(state) {
        const { streamMetadata, playback } = state;

        if (
            !this.shouldSendEvents &&
            streamMetadata.channel &&
            broadcasterLiveStatus(state, streamMetadata.channel.name) !== NOT_AVAILABLE
        ) {
            this.shouldSendEvents = true;
            this.sendQueuedEvents();
        }

        if (!this.hasEventBeenTracked(TRACKING_PAGEVIEW_EVENT) && streamMetadata.url) {
            this.trackPageview();
        }

        if (!this.hasEventBeenTracked(TRACKING_BENCHMARK_VIDEO_STARTED) && playback.hasPlayed) {
            this.trackVideoPlaybackStarted();
        }
    }

    currentTime() {
        return Date.now();
    }

    fetchStartTime() {
        return performanceFetchStartSelector(this.store.getState());
    }

    timeFromFetchStartMillis() {
        const fetchStartTime = this.fetchStartTime();

        return isFinite(fetchStartTime) ? (this.currentTime() - fetchStartTime) : null;
    }

    trackPageview() {
        this.setEventToTracked(TRACKING_PAGEVIEW_EVENT);

        this.sendEvent(TRACKING_PAGEVIEW_EVENT, {
            // eslint-disable-next-line camelcase
            client_time: this.currentTime() / MILLIS_IN_SECOND,
        });
    }

    trackFetchStart() {
        this.setEventToTracked(TRACKING_BENCHMARK_FETCH_START);

        const fetchStartTime = this.fetchStartTime();
        const clientTime = isFinite(fetchStartTime) ? fetchStartTime / MILLIS_IN_SECOND : null;

        this.sendEvent(
            TRACKING_BENCHMARK_FETCH_START,
            {
                // eslint-disable-next-line camelcase
                client_time: clientTime,
            }
        );
    }

    trackAppReady() {
        this.setEventToTracked(TRACKING_BENCHMARK_COMPLETE_TRANSITION);

        this.sendEvent(
            TRACKING_BENCHMARK_COMPLETE_TRANSITION,
            {
                destination: 'clips',
                // eslint-disable-next-line camelcase
                time_from_fetch: this.timeFromFetchStartMillis(),
                // eslint-disable-next-line camelcase
                client_time: this.currentTime() / MILLIS_IN_SECOND,
            }
        );
    }

    trackVideoPlaybackStarted() {
        this.setEventToTracked(TRACKING_BENCHMARK_VIDEO_STARTED);

        this.sendEvent(
            TRACKING_BENCHMARK_VIDEO_STARTED,
            {
                // eslint-disable-next-line camelcase
                client_time: this.currentTime() / MILLIS_IN_SECOND,
            }
        );
    }

    sendEvent(eventName, eventData = {}) {
        if (this.shouldSendEvents) {
            this.store.getState().analyticsTracker.clipsTrackEvent(eventName, eventData);
        } else {
            this.eventsQueue.push({
                eventName,
                eventData,
            });
        }
    }

    sendQueuedEvents() {
        this.eventsQueue.forEach(event => {
            this.store.getState().analyticsTracker.clipsTrackEvent(event.eventName, event.eventData);
        });
        this.eventsQueue = [];
    }
}
