import {
    TRACKING_LOCATION_CLIPS,
    TRACKING_CLIP_FORMAT,
    TRACKING_CLIPS_QOS,
    TRACKING_CLIPS_BUFFER_EMPTY,
    TRACKING_CLIPS_BUFFER_REFILL,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import assign from 'lodash/assign';
import { UIStateSubscriber } from 'ui/state-subscriber';
import { play } from 'actions/video-api';

export const SECONDS_TO_MILLIS_MULT = 1000;
export const CAN_PLAY_CHECK_THRESHOLD = 250;

/*
 * ClipsLoadingTimeSpentTracker is a more strict version of the native buffer empty event.
 * We will be using this to validate playback consistency across the clips-upload player
 * and this new embed player on player-ui.
 */

export class ClipsLoadingTimeSpentTracker extends UIStateSubscriber {
    constructor(store, player) {
        super();

        this.store = store;
        this.bufferEmptyStartTime = null;
        this.videoPlayedOnceThrough = false;
        this.lastCanPlayTime = 0;

        this.canPlayRefill = this.canPlayRefill.bind(this);
        this.canPlayFiredRecently = this.canPlayFiredRecently.bind(this);
        this.trackBufferRefill = this.trackBufferRefill.bind(this);
        this.trackBufferEmpty = this.trackBufferEmpty.bind(this);
        this.handleStateChange = this.handleStateChange.bind(this);

        player.addEventListener('canplay', this.canPlayRefill);
        player.addEventListener('canplaythrough', this.canPlayRefill);
        this.player = player;

        this.subscribe(store, ['playback', 'quality'], (currentState, lastState) => {
            this.handleStateChange(currentState, lastState);
        });
    }

    // eslint-disable-next-line complexity
    handleStateChange({ playback, quality }, { playback: lastPlayback, quality: lastQuality }) {
        if (playback.ended) {
            this.videoPlayedOnceThrough = true;
        }

        const maybeRanOutOfBuffer = playback.hasPlayed && playback.isLoading && !lastPlayback.isLoading;
        const changedQuality = lastQuality.selected !== quality.selected;

        if (!this.videoPlayedOnceThrough && (changedQuality || maybeRanOutOfBuffer)) {
            if (changedQuality) {
                this.trackBufferEmpty();
            } else if (maybeRanOutOfBuffer) {
                this.trackBufferEmptyIfStalled();
            }
        } else if (this.bufferEmptyStartTime && !playback.isLoading) {
            this.trackBufferRefill();
        }
    }

    trackBufferEmptyIfStalled() {
        const bufferStartTime = this.currentTime();

        // This logic protects against seek/waiting events firing buffer empty no matter what
        setTimeout(() => {
            if (!this.canPlayFiredRecently()) {
                this.trackBufferEmpty(bufferStartTime);
            }
        }, CAN_PLAY_CHECK_THRESHOLD);
    }

    canPlayFiredRecently() {
        return Date.now() - this.lastCanPlayTime < CAN_PLAY_CHECK_THRESHOLD;
    }

    canPlayRefill() {
        this.lastCanPlayTime = Date.now();

        if (this.bufferEmptyStartTime) {
            this.trackBufferRefill();
        }
    }

    trackBufferEmpty(bufferStartTime = this.currentTime()) {
        this.bufferEmptyStartTime = bufferStartTime;

        this.store.getState().analyticsTracker.clipsTrackEvent(
            TRACKING_CLIPS_QOS,
            this.bufferTrackingProperties(TRACKING_CLIPS_BUFFER_EMPTY)
        );
    }

    trackBufferRefill() {
        this.store.dispatch(play()); // This mimics the behavior of the clips player for comparison purposes.
        const bufferRefillTimeSeconds = (this.currentTime() - this.bufferEmptyStartTime) / 1000;

        this.store.getState().analyticsTracker.clipsTrackEvent(
            TRACKING_CLIPS_QOS,
            assign(
                {},
                this.bufferTrackingProperties(TRACKING_CLIPS_BUFFER_REFILL),
                // eslint-disable-next-line camelcase
                { buffering_time: bufferRefillTimeSeconds }
            )
        );

        this.bufferEmptyStartTime = null;
    }

    bufferTrackingProperties(type) {
        return {
            backend: TRACKING_LOCATION_CLIPS,
            // eslint-disable-next-line camelcase
            vod_format: TRACKING_CLIP_FORMAT,
            type,
        };
    }

    currentTime() {
        return new Date().getTime();
    }
}
