import sessionStorageJSON from '../util/session-storage-json';
import * as Settings from '../settings';
import { pushScreen, popScreen, STORM_WARNING_SCREEN } from '../actions/screen';
import * as MediaEvents from '../backend/events/media-event';
import { trackEvent } from 'actions/analytics-tracker';

// This module shows a dialog when the user tries to reload the page.
// This is only enabled for large channels x seconds after offline.
// This code should probably be moved to web-client one day.
export function PreventRefreshDialog(player, root, store, options) {
    var channel;
    var viewers;
    var endedTime;

    function init() {
        player.addEventListener(MediaEvents.LOADSTART, onLoadStart);
        player.addEventListener(MediaEvents.ENDED, onEnded);
        window.addEventListener('beforeunload', onBeforeUnload);

        if (player.getChannel()) {
            // Player has already started loading a channel.
            // We started listening for loadstart too late.
            onLoadStart();
        }
    }

    // Get tracking data from sessionStorage if we left a channel.
    function getLeaveData(channel) {
        var allData = sessionStorageJSON.getItem('leaveData');
        if (!allData) {
            return;
        }

        return allData[channel];
    }

    // Set tracking data in sessionStorage when we leave a channel.
    function setLeaveData(channel, data) {
        var allData = sessionStorageJSON.getItem('leaveData');
        if (!allData) {
            allData = {};
        }

        allData[channel] = data;
        sessionStorageJSON.setItem('leaveData', allData);
    }

    // Determine if the user just refreshed the page and track it.
    function detectRefresh() {
        // We only care about live streams.
        if (!channel) {
            return;
        }

        // Fetch the list of ended channels from storage.
        var leaveData = getLeaveData(channel);
        if (!leaveData) {
            return;
        }

        // We showed the dialog for this channel too long ago, ignore it.
        var now = new Date().getTime() / 1000;
        if (leaveData.time + Settings.leaveDialog.refreshTimeout < now) {
            return;
        }

        /* eslint-disable camelcase */
        // The user must have refreshed the page, track it.
        store.dispatch(trackEvent('page_reload', {
            storm_detected: leaveData.stormDetected,
            leave_dialog: leaveData.stormDialog,
            refresh_warning: leaveData.stormWarning,
        }));
        /* eslint-enable camelcase */
    }

    // Determines if we're in a dangerous situation and need to try and
    // prevent users from refreshing the page.
    function detectStorm() {
        if (!channel) {
            // We only care about live channels for now.
            return false;
        }

        if (!player.getEnded()) {
            // Users won't spam refresh if the channel is online.
            return false;
        }

        if (!viewers || viewers < options.leaveDialogViewerThreshold) {
            // We only care about large impact channels.
            return false;
        }

        var now = new Date().getTime() / 1000;
        if (!endedTime || endedTime + Settings.leaveDialog.sinceEnded < now) {
            // Ignore channels that have been offline for too long.
            return false;
        }

        return true;
    }

    function onLoadStart() {
        channel = player.getChannel();
        viewers = null;
        endedTime = null;

        detectRefresh();
    }

    function onEnded() {
        // We have to get the viewers while it's still available.
        viewers = store.getState().viewercount;

        // Store the time when the channel went offline.
        endedTime = new Date().getTime() / 1000;

        var stormDetected = detectStorm();
        if (stormDetected && options.refreshWarningEnabled) {
            showWarning();
        }
    }

    // Fired when the user has hit the exit or refresh button.
    // We get the option of showing a dialog box asking them not to leave.
    function onBeforeUnload(e) {
        var stormDetected = detectStorm();

        // Show the dialog if enabled and there is a storm detected.
        var stormDialog = stormDetected && options.leaveDialogEnabled;

        // Store the current timestamp in the sessionStorage cache.
        // This allows us to detect a page reload and fire an event.
        // We need to store if a storm was detected for tracking reloads.
        setLeaveData(channel, {
            time: new Date().getTime() / 1000,
            stormDetected: stormDetected,
            stormDialog: stormDialog,
            stormWarning: (store.getState().screen === STORM_WARNING_SCREEN),
        });

        if (stormDialog) {
            // Return is required to show the dialog.
            return showDialog(e);
        }
    }

    function showWarning() {
        store.dispatch(pushScreen(STORM_WARNING_SCREEN));

        setTimeout(function() {
            store.dispatch(popScreen());
        }, Settings.leaveDialog.warningDuration * 1000);
    }

    // Show the confirmation dialog when the user tries to leave the page.
    // This must be called from beforeunload in order to work.
    function showDialog(e) {
        // Track the page leave attempt. We can only track leave attempts
        // if we show a dialog or the network request may not work.
        store.dispatch(trackEvent('page_leave_attempt', {
            viewers: viewers,
        }));

        // Fire an event if the user decided to stay on the page.
        //
        // This is hack based on how browers handle beforeunload.
        // There is no way to get the value the user clicked so we have
        // to use a series of timeouts to figure it out.
        //
        // When the dialog is shown, the JS runtime is paused and any
        // timeouts will not be handled until after the dialog.
        // Unfortunately, Chrome fires the timeout relative to when the
        // dialog ends, while Firefox is relative to when the dialog starts.
        //
        // We use one timeout of 1 millisecond so it fires immediately
        // when the dialog is closed across all browsers. But even if the
        // user chooses leave, the page hasn't been unloaded yet. This
        // is an unstable state (network requests may be cancled) which is
        // why we can't track it. We use another timer for 1 second which
        // should give the browser enough time to unload, leaving us with
        // just the users who clicked cancel. The user might also have a
        // really slow PC but there's no way we can tell, so this is good
        // enough.
        setTimeout(function() {
            setTimeout(function() {
                store.dispatch(trackEvent('page_leave_cancel', {
                    viewers: viewers,
                }));
            }, 1000);
        }, 1);

        // eslint-disable-next-line no-param-reassign
        e.returnValue = Settings.leaveDialog.text; // Gecko and Trident
        return Settings.leaveDialog.text; // Gecko and WebKit
    }

    /**
     *
     */
    this.destroy = function() {
        window.removeEventListener('beforeunload', onBeforeUnload);
    };

    init();
}
