import { subscribe } from '../util/subscribe';

const PROPERTY_UNSUBSCRIBES = `unsub_${Date.now()}`;

/**
 * UI parent class to facilitate subscriptions/unsubscriptions to the state store.
 * Derived classes must implement a `handleUpdate` method, which receives the
 * same arguments you would receive from a normal subscriber via the utility
 * function.
 * @abstract
 * @class
 */
export class UIStateSubscriber {
    constructor() {
        this[PROPERTY_UNSUBSCRIBES] = [];
    }

    /**
     * Creates a new subscription to the given store on the given properties,
     * which are cleaned up when the component is destroyed.
     *
     * @param {Redux.Store} store
     *        Store to which the subscription should be registered
     * @param {Array<String>} props
     *        Set of property names which should trigger the callback when changed
     * @param {Function(Object, Object)} fn
     *        Callback function executed when any of the properties change
     */
    subscribe(store, props, fn) {
        this[PROPERTY_UNSUBSCRIBES].push(subscribe(store, props, fn));
    }

    /**
     * Unsubscribes this component from the store.
     */
    destroy() {
        this[PROPERTY_UNSUBSCRIBES].forEach(unsub => unsub());
    }
}
