import assign from 'lodash/assign';

// JavaScript uses 0th based months
export const MONTH = Object.freeze({
    JANUARY: 0,
    FEBRUARY: 1,
    MARCH: 2,
    APRIL: 3,
    MAY: 4,
    JUNE: 5,
    JULY: 6,
    AUGUST: 7,
    SEPTEMBER: 8,
    OCTOBER: 9,
    NOVEMBER: 10,
    DECEMBER: 11,
});

export const NORMAL_YEAR = Object.freeze({
    [MONTH.JANUARY]: 31,
    [MONTH.FEBRUARY]: 28,
    [MONTH.MARCH]: 31,
    [MONTH.APRIL]: 30,
    [MONTH.MAY]: 31,
    [MONTH.JUNE]: 30,
    [MONTH.JULY]: 31,
    [MONTH.AUGUST]: 31,
    [MONTH.SEPTEMBER]: 30,
    [MONTH.OCTOBER]: 31,
    [MONTH.NOVEMBER]: 30,
    [MONTH.DECEMBER]: 31,
});

export const LEAP_YEAR = Object.freeze(assign({}, NORMAL_YEAR, {
    [MONTH.FEBRUARY]: 29,
}));

/**
 * Helper method to assert a given object is a `Date` object.
 *
 * @param {Date} date - the date object to validate
 * @throws {Error} if the date is not an instance of `Date`
 */
export function assertIsDate(date) {
    if (!(date instanceof Date)) {
        throw new Error('Invalid date object');
    }
}

/**
 * Checks if a number is truly an integer.
 *
 * @param {Number} value - the value to test
 * @returns {Boolean} true if the number is an integer, false otherwise
 */
function isInteger(value) {
    return isFinite(value) && Math.round(value) === value;
}

/**
 * Creates a copy of a date object with the hours set to midnight (hh:mm:ss:ms set to 0).
 * This is useful for doing precise date comparisons.
 *
 * @param {Date} date - the date to clear
 * @returns {Date} a date set to exactly midnight
 *
 * @throws {Error} if the date is not an instance of `Date`
 */
export function clearTime(date) {
    const updatedDate = cloneDate(date);
    updatedDate.setHours(0, 0, 0, 0);

    return updatedDate;
}

/**
 * Clones a given date object to prevent modifying the original date.
 *
 * @param {Date} date - the date to clone
 * @returns {Date} a copy of the given date
 *
 * @throws {Error} if the date is not an instance of `Date`
 */
export function cloneDate(date) {
    assertIsDate(date);
    return new Date(date.getTime());
}

/**
 * Determines the maximum number of days for a given month and a year (compensates for leap years).
 *
 * @param {Number} month - the month (0th base index)
 * @param {Number} year - the target year
 * @returns {Number} the maxmum number of days for a given year
 *
 * @throws {Error} if the month is out of range or is not an integer
 * @throws {Error} if the year is not an integer
 */
export function getMaxDays(month, year) {
    if (!isInteger(month) || month < 0 || month > 11) {
        throw new Error(`Invalid month '${month}'. Valid values need to be from 0 (Jan) to 11 (Dec)`);
    }
    return isLeapYear(year) ? LEAP_YEAR[month] : NORMAL_YEAR[month];
}

/**
 * Determines whether or not the input year is a leap year.
 *
 * @param {Number} year - the year to compare
 * @returns {Boolean} true if the input year is a leap year, false otherwise
 *
 * @throws {Error} if the year is not an integer
 */
export function isLeapYear(year) {
    if (!isInteger(year) || year < 0) {
        throw new Error(`Invalid year '${year}'. Value must be an non-negative integer.`);
    }
    return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
}

/**
 * Determines if a given a date is meets the minimum required age (in years).
 *
 * @param {Date} date - the input date (e.g. birthday)
 * @param {Number} minimumAge - the minimum age (in years)
 * @returns {Boolean} true if the taget date meets the required minimum age, false otherwise
 *
 * @throws {Error} if the date is not an instance of `Date`
 * @throws {Error} if the date `minimumAge` is negative, not finite, or not an integer
 */
export function isMinimumAge(date, minimumAge) {
    if (!isInteger(minimumAge) || minimumAge < 0) {
        throw new Error(`Minimum age '${minimumAge}' must be non-negative and an integer`);
    }

    const inputDate = clearTime(date);
    const minimumDate = clearTime(new Date());
    minimumDate.setFullYear(minimumDate.getFullYear() - minimumAge);

    return inputDate.getTime() <= minimumDate.getTime();
}
