import BigScreen from 'bigscreen/bigscreen';
import { setCanFullScreen, setFullScreen } from '../actions/screen-mode';
import { subscribe } from './subscribe';
import { emitFullscreenChange } from '../actions/event-emitter';

export const FULLSCREEN_CHANGE = 'fullscreenchange';

/**
 * Fullscreen Utility
 * Surfaces the functionality for full screening the player.
 * @class
 */
export class FullScreen {
    /**
     * @constructor
     */
    constructor(root, store) {
        this._store = store;
        this._root = root;
        this._store.dispatch(setCanFullScreen(this.canFullScreen()));
        subscribe(this._store, ['screenMode.isFullScreen'], ({ screenMode }) => {
            this.setFullScreen(screenMode.isFullScreen);
        });

        BigScreen.onchange = this._onFullScreenChange.bind(this);
    }

    /**
     * Checks if player is able to fullscreen.
     * @return {Boolean}
     */
    canFullScreen() {
        return BigScreen.enabled || this._checkWebkitFullscreen();
    }

    /**
     * Safari on iOS can only fullscreen if the `webkitSupportsFullscreen` flag
     * on the videotag is true. This flag is only available when `loadedmetadata` has
     * fired.
     *
     * @return {Boolean}
     */
    _checkWebkitFullscreen() {
        const videoTag = this._root.getElementsByTagName('video').item(0);
        if (!videoTag) {
            return false;
        }
        return videoTag.webkitSupportsFullscreen || false;
    }

    /**
     * Checks if player is fullscreen
     * @return {Boolean}
     */
    isFullScreen() {
        return BigScreen.element === this._root;
    }

    /**
     * Sets player to either full screen or not full screen
     * @param {Boolean} isFullScreen - defines if player should be fullscreen
     */
    setFullScreen(isFullScreen) {
        if (!this.canFullScreen()) {
            return;
        }

        if (this.isFullScreen() !== isFullScreen) {
            const cb = this._onFullScreenChange.bind(this);
            BigScreen.toggle(this._root, cb, cb);
        }
    }

    /**
     * Called on any changes to the fullscreen state of the player
     * @private
     */
    _onFullScreenChange() {
        this._store.dispatch(emitFullscreenChange());
        // Hitting escape while in fullscreen mode bypasses the store
        // by invoking a function directly on BigScreen. This line updates
        // store with the current fullscreen state.
        this._store.dispatch(setFullScreen(this.isFullScreen()));
        this._setDeprecatedDataAttrForLegacySystem(this.isFullScreen());
    }

    // Dependency for web-client
    _setDeprecatedDataAttrForLegacySystem(isFullscreen) {
        this._root.dataset.isfullscreen = isFullscreen;
    }
}
