import EventEmitter from 'event-emitter';

export const EVENT_MINUTE = 'minute';
/**
 * Specialized timer for tracking minutes-watched, one of the most important
 * statistics we track.
 */
export function MinutesWatched() {
    // Compute an initial delay for the minutes-watched timer, between 0-60s.
    // The `minutes-watched` event should fire after this timer, as well as
    // every minute thereafter.
    //
    // Since we track minutes-watched with an event, leaving the page
    // early would cause us to undercount. For example, watching a
    // video for 2m40s would only fire 2 minute-watched events. To
    // alleviate this, we want to delay the first minute-watched a random
    // amount between 0s-60s (averages to 30s). So now if you watch a video
    // for 2m40s, 66% of users fire 3 minute-watched and 33% fire 2
    // minute-watched events, averaging our to 2m40s.
    this.initialDelay = Math.round(60 * 1000 * Math.random());
    this._remainingDelay = this.initialDelay;

    // Internally held Timeout for tracking the event.
    this._timeout = null;
    this._timeoutStart = null;

    // Total number of times the minutes-watched event has been fired
    // (approximately equivalent to the number of actual minutes watched)
    this.totalMinutes = 0;

    this._events = new EventEmitter();
}

/**
 * Interval for MinutesWatched timer to fire between tracking events.
 * @const {Number}
 */
MinutesWatched.EVENT_INTERVAL = 60 * 1000;

/**
 * The maximum delay between when the event fires and when it should
 * actually fire. This occurs if the computer stalls, such as falling
 * asleep or using too much CPU. If the delay falls below this number,
 * we adjust the timeout for the next event.
 *
 * For example, if it takes 63 seconds to fire minute-watched, we fire
 * the next event in 57 seconds. But if it takes 85 seconds to fire
 * minute-watched, we fire the next event in 60 seconds like normal.
 *
 * If MAX_DRIFT > EVENT_INTERVAL, then it's possible to fire multiple
 * simultaneous events.
 */
MinutesWatched.MAX_DRIFT = 10 * 1000;

/**
 * Starts the minute-watched timer. Fires an event when a minute has passed.
 * The stop method will pause the timer and continue from where it left off
 * the next time the timer is started.
 */
MinutesWatched.prototype.start = function() {
    if (this._timeout !== null) {
        // Already started, abort.
        return;
    }

    this._scheduleTick();
};

/** Stops the minute-watched timer and records the amount of time remaining.
 * The next time we start the timer, we should delay the remaining amount
 * instead of a full minute.
 */
MinutesWatched.prototype.stop = function() {
    if (this._timeout === null) {
        // Already stopped, abort.
        return;
    }

    // First, make sure the timeout is stopped.
    clearTimeout(this._timeout);
    this._timeout = null;

    // Update the remainingDelay, and fire the event if needed.
    this._tick();
};

// NOTE: This roughly implements the bugged Flash stop behavior.
// TODO: Stop using this once we're comfortable with the numbers.
MinutesWatched.prototype.reset = function() {
    this.stop();

    this._remainingDelay = MinutesWatched.EVENT_INTERVAL;
};

MinutesWatched.prototype.on = function(name, callback) {
    this._events.on(name, callback);
};

MinutesWatched.prototype.off = function(name, callback) {
    this._events.off(name, callback);
};

MinutesWatched.prototype.destroy = function() {
    clearTimeout(this._timeout);
    this._events.removeAllListeners();
};

MinutesWatched.prototype._tick = function() {
    // Determine how long we waited since the timeout was started.
    var now = (new Date()).getTime();
    var elapsed = (now - this._timeoutStart);

    // Subtract the actual elapsed time from the delay.
    // Negative means we waited longer than scheduled (drift),
    // positive means we ended early (stop), and zero means we
    // were completely accurate.
    this._remainingDelay -= elapsed;

    // If the drift is a small amount, we want to schedule the next
    // tick slightly sooner. For example, if remainingDelay is -1s,
    // then we want to schedule the next minute-watched in 59s.
    //
    // If the drift is too large, let's say -3600s, then the system
    // clock went forwards for whatever reason. In that case, just
    // schedule the next minute-watched for 60s like normal.
    if (this._remainingDelay <= -MinutesWatched.MAX_DRIFT) {
        this._remainingDelay = 0;
    }

    // If the remainingDelay is positive and less than EVENT_INTERVAL,
    // it means we stopped the timer before the timeout finished.
    // For example, if remainingDelay is 12s, we don't emit the event
    // and wait for 12s the next time the timer is started.
    //
    // However, remainingDelay is larger than EVENT_INTERVAL, then
    // the clock must have went backwards because we never schedule
    // longer than that. If the clock changed dramatically, then
    // we have no idea how much time was remaining on the timeout
    // when it stopped. We assume they jumped forward an hour and fire
    // the minute event, but it's very possible (but very rare) the
    // timer was stopped in the middle of a clock change.
    if (this._remainingDelay >= MinutesWatched.EVENT_INTERVAL + MinutesWatched.MAX_DRIFT) {
        this._remainingDelay = 0;
    }

    // If the remainingDelay is zero or negative, fire the event
    // and calculate the next delay. We use a while loop just to
    // support MAX_DRIFT > EVENT_INTERVAL, otherwise it behaves the
    // same as an if statement.
    while (this._remainingDelay <= 0) {
        this._remainingDelay += MinutesWatched.EVENT_INTERVAL;
        this.totalMinutes += 1;

        // EventEmitter is synchronous, so this emit might actually take
        // a fair amount of time, which can screw up our timestamps.
        setTimeout(this._events.emit.bind(this._events, EVENT_MINUTE), 0);
    }
};

// Schedule a timeout to fire the next minute-watched event.
MinutesWatched.prototype._scheduleTick = function() {
    this._timeoutStart = (new Date()).getTime();

    var self = this;

    // Start a timeout for the remaining delay.
    this._timeout = setTimeout(function() {
        self._tick();
        self._scheduleTick();
    }, this._remainingDelay);
};

MinutesWatched.prototype.isCurrentlyTracking = function() {
    return Boolean(this._timeout);
};

