import * as Timestamp from './timestamp';

// Parse the query params into an object.
export function parse(query) {
    var params = {};

    var parts = query.split('&');
    for (var i = 0; i < parts.length; i++) {
        // support key=value, key, !key, key=true, key=false
        var result = /^(.+?)(?:=(.+))?$/.exec(parts[i]);
        if (!result) continue;

        var key = result[1];
        var value = result[2];

        // Parse all of the possible options; order matters.
        if (value === 'true') {
            // key=true
            value = true;
        } else if (value === 'false') {
            // key=false
            value = false;
        } else if (value !== undefined) {
            try {
                const parsed = JSON.parse(decodeURIComponent(value));
                value = typeof parsed === 'object' ? parsed : decodeURIComponent(value);
            } catch (e) {
                value = decodeURIComponent(value);
            }
        } else if (key[0] === '!') {
            // !key
            key = key.substring(1);
            value = false;
        } else {
            // key
            value = true;
        }

        params[key] = value;
    }

    return params;
}

// Reverse operation of the above, convert an object into a string.
export function toString(options) {
    var parts = [];

    for (var key in options) {
        if (options.hasOwnProperty(key)) {
            var value = options[key];
            key = encodeURIComponent(key);

            if (value === true) {
                parts.push(key);
            } else if (value === false) {
                parts.push(`!${key}`);
            } else if (value !== null && typeof value === 'object') {
                value = encodeURIComponent(JSON.stringify(value));
                parts.push(`${key}=${value}`);
            } else {
                value = encodeURIComponent(value);
                parts.push(`${key}=${value}`);
            }
        }
    }

    return parts.join('&');
}

/*
 * Url query param time => converting to required format.
 ==============
 * Scenerio 1:
 * expected input = undefined ; when time value is not present in url query.
 * expected output = return same time i.e. undefined.
 ==============
 * Scenerio 2:
 * expected input = in proper format i.e. xxhxxmxxs e.g. '01h20m30s'
 * expected output = return same time
 ==============
 * Scenerio 3:
 * expected input = in string format e.g. '3600'
 * expected output = 01h00m00s; after conversion
 */
export function convertUrlQueryParamTime(time) {
    if (time !== undefined) {
        // Check if time is in proper format i.e. xxhxxmxxs e.g. 02h30m25s
        const regex = /^(?:(\d+)[h])?(?:(\d+)[m])?(?:(\d+)[s])?$/;
        const match = regex.exec(time);

        // if block executes when time value in query url is not in xxhxxmxxs format e.g. '3600'.
        if (!match) {
            const timeInt = parseInt(time,10);
            return Timestamp.toURLString(timeInt);
        }
    }
    return time;
}
