/**
 * Storage wrapper that performs serialization and deserialization of valid JSON
 * values into the provided storage medium.
 */
class Storage {
    /**
     * @param {Storage} storage
     */
    constructor(storage) {
        this._storage = storage;
    }

    /**
     * Retrieves and deserializes the value at the provided key. If no such key
     * exists in the underlying storage, returns the provided `defaultValue`.
     *
     * @param {String} key
     * @param {*} defaultValue
     * @return {*}
     */
    get(key, defaultValue) {
        if (this.has(key)) {
            try {
                return JSON.parse(this._storage.getItem(key));
            } catch (e) {
                // ignore and return the default value
            }
        }

        return defaultValue;
    }

    /**
     * Determines if the underlying storage has a value associated with the
     * given key.
     *
     * @param {String} key
     * @return {Boolean}
     */
    has(key) {
        return (this._storage.getItem(key) !== null);
    }

    /**
     * Serializes and stores the provided value into the underlying storage,
     * associated with the given key.
     *
     * @param {String} key
     * @param {*} value
     */
    set(key, value) {
        this._storage.setItem(key, JSON.stringify(value));
    }

    /**
     * Removes the value associated with the given key from the underlying
     * storage.
     *
     * @param {String} key
     */
    remove(key) {
        this._storage.removeItem(key);
    }

    /**
     * Removes all keys from the underlying storage.
     */
    clear() {
        this._storage.clear();
    }

    usesCache() {
        return !(this._storage instanceof MemoryStorage);
    }
}

/**
 * In-memory implementation of the Storage interface. Used for session-only data
 * and when `localStorage` is not writable (e.g. when in an incognito Safari
 * window).
 *
 * (see https://developer.mozilla.org/en-US/docs/Web/API/Storage)
 *
 * AN: The comments below are straight from the MDN. The wording is known to be
 * awkward.
 */
class MemoryStorage {
    constructor() {
        this._cache = {};
    }

    /**
     * When passed a number n, this method will return the name of the nth key
     * in the storage.
     *
     * @param {Number} key
     *        An integer representing the number of the key you want to get the
     *        name of. This is a zero-based index.
     */
    key(key) {
        return Object.keys(this._cache)[key];
    }

    /**
     * When passed a key name, will return that key's value.
     *
     * @param {String} keyName
     * @return {String}
     */
    getItem(keyName) {
        return (this._cache.hasOwnProperty(keyName) ? this._cache[keyName] : null);
    }

    /**
     * When passed a key name and value, will add that key to the storage, or
     * update that key's value if it already exists.
     *
     * @param {String} keyName
     *        A DOMString containing the name of the key you want to create/update.
     * @param {String} keyValue
     *        A DOMString containing the value you want to give the key you are
     *        creating/updating.
     */
    setItem(keyName, keyValue) {
        this._cache[keyName] = String(keyValue);
    }

    /**
     * When passed a key name, will remove that key from the storage.
     *
     * @param {String} keyName
     *        A DOMString containing the name of the key you want to remove.
     */
    removeItem(key) {
        delete this._cache[key];
    }

    /**
     * When invoked, will empty all keys out of the storage.
     */
    clear() {
        this._cache = {};
    }
}

/**
 * Gets a web storage API, checks for basic API (`getItem`, `setItem`).
 *
 * @params {string} type the type of window.<storage>
 * @returns {boolean} <code>true</code> if the storage looks like a web
 *      storage API, <code>false</code> otherwise.
 */
function getStorage(type) {
    try {
        // FF throws an exception when in incognito mode
        const storage = window[type];
        return verifyStorage(storage);
    } catch (e) {
        // do nothing
    }
    return new MemoryStorage();
}

/**
 * Determines if the provided Storage is suitable for use as a backing store for
 * a Storage (as defined by this module) instance.
 *
 * @param {Storage?} storage
 * @return {Storage}
 */
function verifyStorage(storage) {
    if (storage) {
        const testItem = `test_item_${Date.now()}`;
        try {
            storage.setItem(testItem, 'OK');
            if (storage.getItem(testItem) === 'OK') {
                storage.removeItem(testItem);
                return storage;
            }
        } catch (e) {
            // ignore and use memory storage instead
        }
    }

    return new MemoryStorage();
}

export const sessionStore = new Storage(getStorage('sessionStorage'));
export const localStore = new Storage(getStorage('localStorage'));
export const memoryStore = new Storage(new MemoryStorage());
