import set from 'lodash/set';
import get from 'lodash/get';

/**
 * Subscribes to the given store, and invokes the provided function only when
 * one of the properties listed has changed. The function is given an object
 * with the changed properties as its argument. Returns the unsubscribe function
 * returned by the store.
 *
 * @param {Redux.Store} store
 * @param {Array<String>} props
 * @param {Function} fn
 * @return {Function}
 */
export function subscribe(store, props, fn) {
    let lastState = store.getState();

    return store.subscribe(function() {
        let changed = false;
        const currentState = store.getState();
        const returnedProps = props.reduce((o, key) => {
            set(o, key, get(currentState, key));
            if (get(currentState, key) !== get(lastState, key)) {
                changed = true;
            }

            return o;
        }, {});

        if (changed) {
            const prevState = lastState;
            lastState = currentState;
            fn(returnedProps, prevState);
        }
    });
}
