/**
 * Supports "12h34m56s", "12:34:56", and just "45296".
 *
 * @param {String} str The time to parse.
 * @return {Int}
 */
export function parse(str) {
    var regex = /^(?:(\d+)[h])?(?:(\d+)[m])?(?:(\d+)[s])?$/;
    var match = regex.exec(str);

    if (!match) {
        return null;
    }

    var value = 0;
    for (var i = 1; i < match.length; i++) {
        var num = parseInt(match[i], 10) || 0;
        value = value * 60 + num;
    }

    return value;
}

/**
 * Returns a time broken into array values from.
 *
 * @param {Int} time The time to parse in seconds.
 * @return {Array}
 */
export function toArray(time) {
    var parts = [
        (time / 3600), // hours
        (time / 60) % 60, // minutes
        time % 60, // seconds
    ];

    for (var i = 0; i < parts.length; i++) {
        var rounded = Math.floor(parts[i]);
        parts[i] = (rounded < 10) ? `0${rounded}` : rounded;
    }

    return parts;
}

function cleanString(timeString) {
    for (var i = 0; i < timeString.length; i++) {
        if (timeString[i] !== '0' && timeString[i] !== ':') {
            return timeString.substring(i, timeString.length);
        }
    }

    return '0';
}

/**
 * Returns a time in seconds in the format "xx:yy:zz"
 *
 * @param {Int} time The time to parse in seconds.
 * @param {Boolean} clean If the array should remove values of 0.
 * @return {String}
 */
export function toString(time, clean) {
    if (!isFinite(time) || time < 0) {
        return '00:00:00';
    }

    const parts = toArray(time);
    const timeString = parts.join(':');

    return clean ? cleanString(timeString) : timeString;
}

/**
 * Returns a time in seconds as a URL ready string. "11h22m33s"
 *
 * @param {Int} time The time to parse in seconds.
 * @return {String}
 */
export function toURLString(time) {
    var parts = toArray(time);
    var times = ['h', 'm', 's'];
    var output = '';

    parts = parts.reverse();
    times = times.reverse();
    for (var i = 0; i < parts.length; i++) {
        output = parts[i] + times[i] + output;
    }

    return output.replace(/00(h|m|s)/g, '');
}

/**
 * Returns a string in 'yyyy-mm-dd' date format.
 *
 * @param {Date} date Arbitrary Date object.
 * @param {Boolean} zeroPad If month and date should be prepended with 0 for double digit format.
 * @return {String}
 */
export function toDateString(date, zeroPad) {
    const year = date.getFullYear();
    let month = date.getMonth() + 1;
    let day = date.getDate();
    if (zeroPad) {
        month = month < 10 ? `0${month}` : month;
        day = day < 10 ? `0${day}` : day;
    }
    return `${year}-${month}-${day}`;
}
