import assign from 'lodash/assign';
import { stringify } from 'query-string';
import { apiHost as API_HOST } from '../settings';

/**
 * Fetches the access token from the given URL, using the provided OAuth token.
 *
 * @param {String} url
 * @param {?String} oauthToken
 * @param {Object} params
 * @return {Promise<Object>}
 */
export function fetchAccessToken(url, oauthToken, params) {
    const data = assign({}, params, {
        oauth_token: oauthToken, // eslint-disable-line camelcase
    });

    return fetch(`${url}?${stringify(data)}`, {
        credentials: 'include',
        headers: {
            'Client-ID': 'jzkbprff40iqj646a697cyrvl0zt2m6',
        },
    }).then(response => response.json());
}

/**
 * Fetches the access token (which contains the OAuthToken) for a VOD stream from the API.
 *
 * @param {String} videoId canonical video ID to which the token applies
 * @param {?String} oauthToken OAuth token for the user
 * @param {Object} params additional parameters for the nauth endpoint
 * @return {Promise<Object>}
 */
export function fetchVideoAccessToken(videoId, oauthToken, params) {
    const url = `${API_HOST}/api/vods/${videoId.substring(1)}/access_token`;

    return fetchAccessToken(url, oauthToken, params);
}

/**
 * Fetches the access token (which contains the OAuthToken) for a live stream from the API.
 *
 * @param {String} channelName canonical channel name to which the token applies
 * @param {?String} oauthToken OAuth token for the user
 * @param {Object} params additional parameters for the nauth endpoint
 * @return {Promise<Object>}
 */
export function fetchChannelAccessToken(channelName, oauthToken, params) {
    const url = `${API_HOST}/api/channels/${channelName.toLowerCase()}/access_token`;

    return fetchAccessToken(url, oauthToken, params);
}
