import EventEmitter from 'event-emitter';
import { ExtensionCoordinator } from 'extension-coordinator';
import { BACKEND_PLAYER_CORE } from './backend/player-core';
import { BackendChromecast } from './backend/chromecast';
import { BackendMulti } from './backend/multi';
import { BackendBlank, BACKEND_BLANK_TYPE } from './backend/blank';
import { localStore } from 'util/storage';
import * as Settings from './settings';
import { CONTENT_MODE_LIVE } from './stream/twitch-live';
import { CONTENT_MODE_VOD } from './stream/twitch-vod';
import { CONTENT_MODE_CLIP } from './stream/clip';
import { subscribe } from './util/subscribe';
import { setCaptionsData, ACTION_TOGGLE_CAPTIONS } from './actions/captions';
import { setBackend } from './actions/backend';
import { setPlayerBackendType } from './actions/backend-info';
import { requestAds, playAd, pauseAd,
         setAdClickThrough, DEFAULT_AD_DURATION, setCurrentAdMetadata,
         clearCurrentAdMetadata, AdContentTypes, AdRollTypes } from './actions/ads';
import { PREROLL, MIDROLL, POSTROLL } from './ads/ima-tags';
import { createAdsManager } from './actions/ads-manager';
import { nullAdsManager } from './ads/null-manager';
import { incrementQualityChangeCount } from './actions/analytics';
import { trackEvent } from './actions/analytics-tracker';
import { setManifestInfo } from './actions/manifest-info';
import { setAccessTokenParams } from './actions/access-token';
import { setCastingState, setDeviceName } from './actions/chromecast';
import { setError, clearError } from './actions/error';
import { setOnline } from './actions/online';
import { playerMuted, volumeChanged, updateDuration,
         setLoading, updatePlaybackState, updateCurrentTime,
         updateBufferValues, playbackRateChanged, incrementBufferEmpties,
         playerSeeking, playerSeeked } from './actions/playback';
import { selectQuality, setCurrentQuality, setQualities,
         QUALITY_AUTO, QUALITY_AUTO_OBJECT, KEY_AUTO_QUALITY_FORCED,
         setPreferredQuality, DEFAULT_STREAM_FORMAT, DEFAULT_STREAM_BITRATE_IN_BPS,
         setABSAvailability } from './actions/quality';
import { fetchLiveStreamMetadata } from './actions/stream-metadata';
import { pushScreen, popScreen, ADVERTISEMENT_SCREEN } from './actions/screen';
import { setFullScreen, setTheatreMode } from './actions/screen-mode';
import { setUsherParams } from './actions/usher';
import { videoAPILoaded } from './actions/video-api';
import { eventEmitterLoaded } from 'actions/event-emitter';
import defaults from 'lodash/defaults';
import extend from 'lodash/extend';
import includes from 'lodash/includes';
import isString from 'lodash/isString';
import isFunction from 'lodash/isFunction';
import each from 'lodash/each';
import * as MediaEvents from './backend/events/media-event';
import * as TwitchEvents from './backend/events/twitch-event';
import * as AdEvents from './ads/advertisement-event';
import { updateStats, STATS_UPDATE_INTERVAL } from './actions/stats';
import { SITE_MINIPLAYER_ACTION } from './analytics/analytics';
import { ONLINE_STATUS } from './state/online-status';
import { PLAYBACK_ERROR } from './analytics/spade-events';
import { BACKEND_MAP, getAvailableBackends } from './backend/util';
import { SEEKABLE_TYPES } from 'util/seekable-types';
import Errors from 'errors';

export const ID3_AD = 'ID3_AD';

export function Video(root, store, options) {
    var self = this;

    const events = new EventEmitter();
    store.dispatch(eventEmitterLoaded(events));

    const unsubscribes = [];

    // Private variables.
    let backend;
    let chromecast;
    let unsubscribeSetupAdsManager;

    function init() {
        const availableBackends = getAvailableBackends(store.getState().env.playerType);
        let backendName = BACKEND_BLANK_TYPE;

        if (availableBackends.length < 1) {
            store.dispatch(setError(Errors.CODES.RENDERER_NOT_AVAILABLE));
            store.dispatch(trackEvent(PLAYBACK_ERROR, {
                // eslint-disable-next-line camelcase
                playback_error_code: 6000,
                // eslint-disable-next-line camelcase
                playback_error_msg: 'no_backend_supported',
            }));
        } else if (includes(availableBackends, options.backend)) {
            backendName = options.backend;
        } else {
            backendName = availableBackends[0];
        }

        loadBackend(backendName);
        ExtensionCoordinator.ExtensionService.registerPlayer(self);
        ExtensionCoordinator.ExtensionService.listenForContext();
        _initStatsLoop();
    }

    // eslint-disable-next-line max-statements
    function loadBackend(backendName) {
        // Create a div for the player so we can start loading ASAP.
        const videoContainer = document.createElement('div');
        videoContainer.className = 'player-video';
        root.appendChild(videoContainer);

        // set up nauth access token params
        const { env, ads } = store.getState();
        store.dispatch(setAccessTokenParams({
            // eslint-disable-next-line camelcase
            adblock: ads.adblock,
            // eslint-disable-next-line camelcase
            need_https: backendName === BACKEND_PLAYER_CORE,
            platform: env.platform,
            // eslint-disable-next-line camelcase
            player_type: env.playerType,
        }));
        const playerBackend = backendName;
        store.dispatch(setUsherParams(playerBackend, options));

        var BackendType = BACKEND_MAP[backendName] || BackendBlank;

        // Add in the lastAdDisplay to the options.
        // TODO Refactor this into an advertisements class.
        var backendOpts = extend({
            lastAdDisplay: localStore.get('lastAdDisplay', 0),
        }, options);

        // Create the main backend.
        const mainBackend = new BackendType(backendOpts, store);
        store.dispatch(setBackend(mainBackend));
        store.dispatch(setPlayerBackendType(mainBackend.getBackend()));
        mainBackend.attach(videoContainer);

        // Backend used to connect to Chromecast.
        chromecast = BackendChromecast;
        const castingState = chromecast.init();
        // Restore existing casting state into redux
        store.dispatch(setCastingState(castingState));

        // Use a multi backend to switch between chromecast and main.
        backend = new BackendMulti(chromecast, mainBackend);

        _checkIMAScriptStatus(videoContainer, backend);

        // Doing it during initEvents affects UI pause/play/etc.
        // ie: no longer responsive. Can probably work around if need to.
        mainBackend.addEventListener(MediaEvents.LOADED_METADATA, onLoadedMetadata);
        chromecast.addEventListener(MediaEvents.LOADED_METADATA, onLoadedMetadata);
        mainBackend.addEventListener(MediaEvents.DURATION_CHANGE, onDurationChange);
        mainBackend.addEventListener(TwitchEvents.BUFFER_CHANGE, onBufferChange);
        mainBackend.addEventListener(TwitchEvents.MIDROLL_REQUESTED, onMidrollRequested);
        mainBackend.addEventListener(TwitchEvents.STITCHED_AD_START, onStitchedAdStart);
        mainBackend.addEventListener(TwitchEvents.STITCHED_AD_END, onStitchedAdEnd);
        mainBackend.addEventListener(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, onABSStreamFormatChange);
        mainBackend.addEventListener(TwitchEvents.OFFLINE, onOffline);
        mainBackend.addEventListener(MediaEvents.ENDED, onEnded);
        chromecast.addEventListener(MediaEvents.ENDED, onChromecastEnded);
        mainBackend.addEventListener(MediaEvents.PLAYING, onPlaying);
        mainBackend.addEventListener(MediaEvents.TIME_UPDATE, onTimeUpdate);
        chromecast.addEventListener(MediaEvents.TIME_UPDATE, onTimeUpdate);
        mainBackend.addEventListener(MediaEvents.RATE_CHANGE, onRateChange);
        // Browsers like Safari and Edge fail to fire playing event on seek/resume content.
        // Using timeupdate event as fallback to avoid loading spinner while content plays fine
        // TODO No longer needed once player-core supports HTMLVideoElement semantics
        const userAgent = store.getState().window.navigator.userAgent.toLowerCase();
        const isSafari = userAgent.indexOf('safari') > -1 &&
                        userAgent.indexOf('chrome') === -1;
        const isEdge = userAgent.indexOf('windows') > -1 &&
                        userAgent.indexOf('edge') > -1;

        if (isSafari || isEdge) {
            mainBackend.addEventListener(MediaEvents.TIME_UPDATE, () => {
                store.dispatch(setLoading(false));
            });
        }

        mainBackend.addEventListener(TwitchEvents.PLAYBACK_STATISTICS, onPlaybackStats);
        mainBackend.addEventListener(MediaEvents.SEEKED, onSeeked);
        mainBackend.addEventListener(MediaEvents.SEEKING, onWaiting);
        mainBackend.addEventListener(MediaEvents.WAITING, onWaiting);
        mainBackend.addEventListener(MediaEvents.WAITING, captureBufferEmpties);
        mainBackend.addEventListener(MediaEvents.PAUSE, onPause);
        mainBackend.addEventListener(TwitchEvents.PLAYER_INIT, self._loadVideoAPI);
        mainBackend.addEventListener(MediaEvents.LOADSTART, self._loadVideoAPI);

        const { playback } = store.getState();
        mainBackend.setVolume(playback.volume);
        mainBackend.setMuted(playback.muted);

        initEvents();
        self._propagateBackendMutliEvents();
    }

    // Load video api into middleware
    self._loadVideoAPI = function() {
        store.dispatch(videoAPILoaded(self));
    };

    // Explicitly propagate events from the backend
    self._propagateBackendMutliEvents = function() {
        const backendMultiEvents = [
            MediaEvents.DURATION_CHANGE,
            TwitchEvents.PLAYER_INIT,
            MediaEvents.LOADSTART,
            MediaEvents.LOADED_METADATA,
            MediaEvents.PLAY,
            MediaEvents.PAUSE,
            MediaEvents.WAITING,
            MediaEvents.PLAYING,
            MediaEvents.ENDED,
            MediaEvents.SEEKING,
            MediaEvents.TIME_UPDATE,
            MediaEvents.CAN_PLAY,
            TwitchEvents.SEGMENT_CHANGE,
            TwitchEvents.BUFFER_CHANGE,
            TwitchEvents.RESTRICTED,
            MediaEvents.SEEKED,
            TwitchEvents.STITCHED_AD_END,
        ];

        backendMultiEvents.forEach(event => {
            backend.addEventListener(event, payload => {
                events.emit(event, payload);
            });
        });
    };

    self._setABS = function _setABS() {
        const forceABS = options.abs;
        const isFirstTime = (!localStore.get(KEY_AUTO_QUALITY_FORCED, false));
        const isAuto = (store.getState().quality.selected === QUALITY_AUTO);

        if (self.absAvailable() && (isFirstTime || forceABS)) {
            store.dispatch(selectQuality(QUALITY_AUTO));
            store.dispatch(setPreferredQuality(QUALITY_AUTO_OBJECT));
            localStore.set(KEY_AUTO_QUALITY_FORCED, true);
        } else if (!self.absAvailable() && isAuto) {
            store.dispatch(selectQuality(DEFAULT_STREAM_FORMAT, DEFAULT_STREAM_BITRATE_IN_BPS));
        }

        store.dispatch(setABSAvailability(self.absAvailable()));
    };

    self.absAvailable = function() {
        return backend.absAvailable();
    };

    function onABSStreamFormatChange(qualityChangeStats) {
        /* eslint-disable camelcase */
        const { stream_format_current, stream_format_previous } = qualityChangeStats;
        if (stream_format_previous !== stream_format_current) {
            events.emit(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, qualityChangeStats);
        }
        /* eslint-enable camelcase */
    }

    function onPlaybackStats(stats) {
        store.dispatch(updateStats(stats));
    }

    function onDurationChange() {
        store.dispatch(updateDuration(store.getState().backend.getDuration()));
    }

    function onWaiting() {
        store.dispatch(setLoading(true));
        store.dispatch(updatePlaybackState(MediaEvents.WAITING));
    }

    function captureBufferEmpties() {
        if (!self.getSeeking()) {
            store.dispatch(incrementBufferEmpties());
        }
    }

    function onLoadedMetadata() {
        // TODO: Remove this hack, explained and tracked in VP-1976.
        store.dispatch({
            type: ACTION_TOGGLE_CAPTIONS,
            captions: {
                data: null,
                available: false,
            },
        });

        store.dispatch(setQualities(self.getQualities()));

        const { quality, stream, playback } = store.getState();

        // check if the default quality chosen is restricted, which we only know
        // after the stream metadata loads
        const allowedQualities = quality.available.filter(q => !includes(stream.restrictedBitrates, q.group));
        if (allowedQualities.length > 0 && !allowedQualities.some(q => q.group === quality.selected)) {
            store.dispatch(selectQuality(allowedQualities[0].group, allowedQualities[0].bandwidth));
        }

        store.dispatch(updateDuration(store.getState().backend.getDuration()));

        if (includes(SEEKABLE_TYPES, stream.contentType) && playback.startTimeSet) {
            self.setCurrentTime(playback.startTime);
        }

        if (stream.contentType === CONTENT_MODE_LIVE) {
            store.dispatch(setOnline(true));
        }

        if (
            options.autoplay &&
            (stream.contentType === CONTENT_MODE_LIVE || stream.contentType === CONTENT_MODE_VOD)
        ) {
            _dispatchPreroll();
        }

        if (stream.contentType === CONTENT_MODE_LIVE) {
            store.dispatch(fetchLiveStreamMetadata(backend.getVideoInfo().broadcast_id));
        }
    }

    function onBufferChange(data) {
        store.dispatch(updateBufferValues(data.start, data.end));
    }

    function onMidrollRequested(data) {
        store.dispatch(requestAds(MIDROLL, data.duration, false, 0, ID3_AD));
    }

    function onStitchedAdStart(data) {
        if (data.hasOwnProperty('URL') && isString(data.URL)) {
            const adClickThroughUrl = data.URL;
            try {
                store.dispatch(setAdClickThrough(adClickThroughUrl));
            } catch (e) {
                // eslint-disable-next-line no-console
                console.warn('Failed to decode click-through URL: ', adClickThroughUrl);
            }
        }
        store.dispatch(setCurrentAdMetadata({
            contentType: AdContentTypes.STITCHED,
            rollType: AdRollTypes.MIDROLL,
        }));
    }

    function onStitchedAdEnd() {
        store.dispatch(clearCurrentAdMetadata());
    }

    function onPlaying() {
        // TEMPORARY: Currently errors from backend are recoverable, so
        // clear error on stream resuming playback.
        store.dispatch(clearError());
        store.dispatch(setLoading(false));
        store.dispatch(updatePlaybackState(MediaEvents.PLAYING));
    }

    function onPause() {
        const { ads } = store.getState();
        // Unfortunately, AD_START gets fired before onContentPauseRequested and since
        // onContentPauseRequested pauses the backend for VODs, we want to ignore this signal
        // so the UI is reflecting the proper playback state.
        if (ads.currentMetadata.contentType !== AdContentTypes.IMA) {
            store.dispatch(updatePlaybackState(MediaEvents.PAUSE));
        }
    }

    function onTimeUpdate() {
        const currentTime = self.getCurrentTime(); // both of these are in seconds
        store.dispatch(updateCurrentTime(currentTime));
    }

    function onRateChange() {
        store.dispatch(playbackRateChanged(backend.getPlaybackRate()));
    }

    function onSeeked() {
        const currentTime = self.getCurrentTime(); // both of these are in seconds
        store.dispatch(playerSeeked(currentTime));
    }

    function onEnded() {
        store.dispatch(updatePlaybackState(MediaEvents.ENDED));
        const { stream, playback } = store.getState();
        if (
            stream.contentType === CONTENT_MODE_VOD ||
            !playback.hasPlayed
        ) {
            return;
        }

        store.dispatch(requestAds(POSTROLL, DEFAULT_AD_DURATION));
    }

    function onChromecastEnded() {
        store.dispatch(updatePlaybackState(MediaEvents.ENDED));
        const { stream } = store.getState();
        if (stream.contentType === CONTENT_MODE_VOD) {
            backend.setCurrentTime(chromecast.getCurrentTime());
        }
    }

    function onOffline() {
        store.dispatch(setOnline(false));
    }

    function requestFirstAd() {
        const { playerOptions } = store.getState();
        if (playerOptions.force_preroll) {
            forceRequestAd(PREROLL, playerOptions.force_preroll);
        } else if (playerOptions.force_midroll) {
            forceRequestAd(MIDROLL, playerOptions.force_midroll);
        } else if (playerOptions.force_preroll_id) {
            forceRequestAd(PREROLL, DEFAULT_AD_DURATION, playerOptions.force_preroll_id);
        } else if (playerOptions.force_midroll_id) {
            forceRequestAd(MIDROLL, DEFAULT_AD_DURATION, playerOptions.force_midroll_id);
        } else {
            store.dispatch(requestAds(PREROLL, DEFAULT_AD_DURATION));
        }
    }

    function _dispatchPreroll() {
        const { adsManager } = store.getState();
        if (adsManager !== nullAdsManager) {
            requestFirstAd();
        } else {
            const unsubscribe = subscribe(store, ['adsManager'], function({ adsManager }) {
                if (adsManager !== nullAdsManager) {
                    unsubscribe();
                    requestFirstAd();
                }
            });
        }
    }

    function _subscribeStream() {
        return subscribe(store, ['stream'], function({ stream }) {
            self._updateStream(stream);
        });
    }

    function _subscribeViewercount() {
        return subscribe(store, ['viewercount'], function() {
            events.emit(TwitchEvents.VIEWERS_CHANGE);
        });
    }

    function _subscribeOnlineStatus() {
        return subscribe(store, ['onlineStatus'], function({ onlineStatus }) {
            if (onlineStatus === ONLINE_STATUS) {
                events.emit(TwitchEvents.ONLINE);
            } else {
                events.emit(TwitchEvents.OFFLINE);
            }
        });
    }

    function _subscribeReadyPlayback() {
        return subscribe(store, ['playback.contentShowing'], function({ playback }) {
            if (playback.contentShowing) {
                events.emit(TwitchEvents.CONTENT_SHOWING);
            }
        });
    }

    function _subscribeQuality() {
        return subscribe(store, ['quality.selected'], function({ quality }) {
            self.setQuality(quality.selected);
        });
    }

    function _subscribeAdsManager() {
        _addAdsManagerListeners(store.getState());
        return subscribe(store, ['adsManager'], _addAdsManagerListeners);
    }

    function _checkIMAScriptStatus(videoContainer, backend) {
        if (store.getState().ads.imaScriptLoaded === true) {
            _setupAdsManager(videoContainer, backend);
        } else {
            unsubscribeSetupAdsManager = subscribe(store, ['ads.imaScriptLoaded'], () => {
                _setupAdsManager(videoContainer, backend);
            });
        }
    }

    function _setupAdsManager(videoContainer, mainBackend) {
        const { ads } = store.getState();
        if (ads.imaScriptLoaded === true) {
            store.dispatch(createAdsManager(videoContainer, mainBackend, store, options));
            unsubscribes.push(_subscribeAdsManager());
            if (isFunction(unsubscribeSetupAdsManager)) {
                unsubscribeSetupAdsManager();
            }
        }
    }

    function _addAdsManagerListeners({ adsManager }) {
        adsManager.addEventListener(AdEvents.AD_END, function(evt) {
            localStore.set('lastAdDisplay', (new Date()).getTime());
            if (evt.roll_type === POSTROLL) {
                store.dispatch(updatePlaybackState(MediaEvents.ENDED));
            }
        });

        adsManager.addEventListener(AdEvents.AD_START, function() {
            store.dispatch(updatePlaybackState(MediaEvents.PLAYING));
        });

        adsManager.addEventListener(AdEvents.AD_ERROR, function(evt) {
            if (evt.roll_type === POSTROLL) {
                store.dispatch(updatePlaybackState(MediaEvents.ENDED));
            }
        });
        // Eliminate the race condition where a module listens to an ad event
        // before an adManager is instantiated.
        each(AdEvents, adEvent => {
            adsManager.addEventListener(adEvent, data => {
                events.emit(adEvent, data);
            });
        });
    }

    function _statsUpdate() {
        const stats = backend.getStats();
        store.dispatch(updateStats(stats));
        events.emit(TwitchEvents.STATS_UPDATE);
    }

    function _initStatsLoop() {
        const { window: windowObj } = store.getState();
        if (!self._statsLoop) {
            self._statsLoop = windowObj.setInterval(_statsUpdate, STATS_UPDATE_INTERVAL);
            unsubscribes.push(() => {
                windowObj.clearInterval(self.statsLoop);
                self._statsLoop = null;
            });
        }
    }

    function initEvents() {
        backend.addEventListener(AdEvents.AD_START, function() {
            store.dispatch(pushScreen(ADVERTISEMENT_SCREEN));
        });
        backend.addEventListener(AdEvents.AD_END, function() {
            localStore.set('lastAdDisplay', (new Date()).getTime());
            store.dispatch(popScreen());
        });

        chromecast.addEventListener(TwitchEvents.CASTING_CHANGE, castingState => {
            store.dispatch(setCastingState(castingState));
            const deviceName = isString(chromecast.getDevice())
                ? chromecast.getDevice()
                : 'Chromecast';
            store.dispatch(setDeviceName(deviceName));
        });

        unsubscribes.push(_subscribeStream());
        unsubscribes.push(_subscribeViewercount());
        unsubscribes.push(_subscribeOnlineStatus());
        unsubscribes.push(_subscribeReadyPlayback());
        unsubscribes.push(_subscribeQuality());

        backend.addEventListener(TwitchEvents.CAPTION_UPDATE, function() {
            store.dispatch(setCaptionsData(backend.getCaption()));
        });

        backend.addEventListener(TwitchEvents.QUALITY_CHANGE, function({ quality, isAuto }) {
            store.dispatch(setCurrentQuality(quality, isAuto));
            store.dispatch(incrementQualityChangeCount());
            store.dispatch(setQualities(self.getQualities()));
        });

        backend.addEventListener(TwitchEvents.MANIFEST_EXTRA_INFO, function(extraInfo) {
            store.dispatch(setManifestInfo(extraInfo));
            store.dispatch(setLoading(true));
            self._initQuality();
        });
    }

    self._initQuality = function() {
        self._setABS();
        setSelectedQuality();
    };

    // needed to ensure we always set the correct quality.
    function setSelectedQuality() {
        const { quality } = store.getState();
        self.setQuality(quality.selected);
    }

    /**
     * Used to track the `quality_change_complete` event, which fires when two segments occur in a row
     * without a quality change in between after a stream is loaded.
     */
    function trackTimeToStableQuality() {
        function playingHandler() {
            backend.removeEventListener(MediaEvents.PLAYING, playingHandler);

            const startTime = Date.now();
            let firstSegmentReceived = false;

            function formatChangedHandler() {
                firstSegmentReceived = false;
            }

            function segmentChangeHandler() {
                if (firstSegmentReceived) {
                    const timeToStable = Date.now() - startTime;
                    // eslint-disable-next-line camelcase
                    const { serving_id } = store.getState().manifestInfo;
                    store.dispatch(trackEvent('quality_change_complete', {
                        /* eslint-disable camelcase */
                        time_to_stable_quality: timeToStable,
                        serving_id: serving_id,
                        /* eslint-enable camelcase */
                    }));
                    backend.removeEventListener(TwitchEvents.SEGMENT_CHANGE, segmentChangeHandler);
                    backend.removeEventListener(TwitchEvents.QUALITY_CHANGE, formatChangedHandler);
                } else {
                    firstSegmentReceived = true;
                }
            }

            backend.addEventListener(TwitchEvents.SEGMENT_CHANGE, segmentChangeHandler);
            backend.addEventListener(TwitchEvents.QUALITY_CHANGE, formatChangedHandler);
        }

        backend.addEventListener(MediaEvents.PLAYING, playingHandler);
    }

    function forceRequestAd(adType, duration, creativeId = 0) {
        const modifiedDuration = parseInt(duration, 10) || DEFAULT_AD_DURATION;
        store.dispatch(requestAds(adType, modifiedDuration, true, creativeId));
    }

    self.destroy = function() {
        backend.destroy();
        unsubscribes.forEach(unsub => unsub());
        ExtensionCoordinator.ExtensionService.unregisterPlayer(self);

        // TODO Clean up any remaining listeners we missed.
        events.removeAllListeners();
    };

    self._updateStream = function(stream) {
        if (stream.contentType === CONTENT_MODE_LIVE) {
            backend.setChannel(stream.channel, stream);
            trackTimeToStableQuality();
        } else if (stream.contentType === CONTENT_MODE_VOD) {
            backend.setVideo(stream.videoId, stream);
            trackTimeToStableQuality();
        } else if (stream.contentType === CONTENT_MODE_CLIP) {
            const { quality } = store.getState();
            // the streamUrl is based on the selected quality
            backend.setQuality(quality.selected);
        } else {
            stream.streamUrl.then(streamUrl => {
                backend.setSrc(streamUrl);
            });
        }
    };

    self.addEventListener = function(name, callback) {
        if (!includes(Settings.allEvents, name)) {
            // eslint-disable-next-line no-console
            console.error('subscribing to unknown event: ', name);
        }

        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        events.off(name, callback);
    };

    self.getNetworkProfile = function() {
        return store.getState().backend.getNetworkProfile();
    };

    // IE8 does not support getters/setters so there are two APIs.
    // Use the function prefixed with get/set for IE8 support, or use
    // standard getters/setters to be compatible with the media tag.

    self.getError = function() {
        return backend.getError();
    };

    self.getSrc = function() {
        // unimplemented
    };

    self.setSrc = function() {
        // unimplemented
    };

    self.getNetworkState = function() {
        return backend.getNetworkState();
    };

    self.getBuffered = function() {
        return backend.getBuffered();
    };

    self.load = function() {
        backend.load();
    };

    self.getReadyState = function() {
        return backend.getReadyState();
    };

    self.getSeeking = function() {
        return backend.getSeeking();
    };

    self.getCurrentTime = function() {
        return backend.getCurrentTime();
    };

    self.setCurrentTime = function(value) {
        store.dispatch(playerSeeking());
        backend.setCurrentTime(value);
    };

    self.getDuration = function() {
        return backend.getDuration();
    };

    self.getPaused = function() {
        const { ads, adsManager } = store.getState();
        if (ads.currentMetadata.contentType === AdContentTypes.IMA) {
            return adsManager.paused;
        }
        return backend.getPaused();
    };

    /** alias for getPaused to match the EmbedPlayer API */
    self.isPaused = self.getPaused;

    self.getPlaybackRate = function() {
        return backend.getPlaybackRate();
    };

    self.setPlaybackRate = function(value) {
        backend.setPlaybackRate(value);
    };

    self.getPlayed = function() {
        return backend.getPlayed();
    };

    self.getEnded = function() {
        return backend.getEnded();
    };

    self.getAutoplay = function() {
        return options.autoplay;
    };

    self.setLoop = function(value) {
        backend.setLoop(value);
    };

    self.play = function() {
        const { playback, ads, stream } = store.getState();
        if (
            !playback.hasPlayed &&
            (stream.contentType === CONTENT_MODE_VOD || stream.contentType === CONTENT_MODE_LIVE)
        ) {
            _dispatchPreroll();
        }
        store.dispatch(updatePlaybackState(MediaEvents.PLAYING));
        if (backend.getNetworkState === backend.NETWORK_EMPTY) {
            backend.load();
        } else if (ads.currentMetadata.contentType === AdContentTypes.IMA) {
            store.dispatch(playAd());
        } else {
            if (stream.contentType === CONTENT_MODE_VOD && self.getEnded()) {
                self.setCurrentTime(0);
            }
            backend.play();
        }
        store.dispatch(trackEvent('video_pause', {
            action: 'play',
            // eslint-disable-next-line camelcase
            in_ad: store.getState().ads.currentMetadata.contentType !== AdContentTypes.NONE,
        }));
    };

    self.pause = function() {
        store.dispatch(updatePlaybackState(MediaEvents.PAUSE));
        const { ads } = store.getState();
        if (ads.currentMetadata.contentType === AdContentTypes.IMA) {
            store.dispatch(pauseAd());
        } else {
            backend.pause();
        }
        store.dispatch(trackEvent('video_pause', {
            action: 'pause',
            // eslint-disable-next-line camelcase
            in_ad: store.getState().ads.currentMetadata.contentType !== AdContentTypes.NONE,
        }));
    };

    self.automatedPause = function() {
        backend.pause();
    };

    self.getControls = function() {
        // TODO Support disabling controls.
        return true;
    };

    self.setControls = function() {
        // TODO Support disabling controls.
        // unimplemented
    };

    self.getVolume = function() {
        return store.getState().playback.volume;
    };

    self.setVolume = function(volume) {
        const { ads, adsManager } = store.getState();
        const volumeRecipient = ads.currentMetadata.contentType === AdContentTypes.IMA ? adsManager : backend;
        volumeRecipient.setVolume(volume);

        store.dispatch(volumeChanged(volume));

        // Save the volume in local storage for next load.
        localStore.set('volume', volume);
    };

    self.getMuted = function() {
        return store.getState().playback.muted;
    };

    /**
     * Set audio muted state on player.
     *
     * @param {Boolean} muted Whether audio should be muted or not.
     * @param {Boolean} automated Whether it comes from automated logic, not from user interaction.
     */
    self.setMuted = function(muted, automated) {
        const { ads, adsManager } = store.getState();
        const volumeRecipient = ads.currentMetadata.contentType === AdContentTypes.IMA ? adsManager : backend;
        volumeRecipient.setMuted(muted);

        store.dispatch(playerMuted(muted));

        if (!automated) {
            // Save the muted state in local storage for next load.
            localStore.set('muted', muted);
        }
    };

    self.getTheatre = function() {
        return store.getState().screenMode.isTheatreMode;
    };

    self.setTheatre = function(value) {
        store.dispatch(setTheatreMode(value));
    };

    /**
     * getQuality returns the quality that CAN be selected by the user
     * and NOT necessarily the variant that is currently playing
     *
     * @returns {String}
     */
    self.getQuality = function() {
        return store.getState().playback.quality;
    };

    self.setQuality = function(quality) {
        backend.setQuality(quality);
    };

    self.getQualities = function() {
        return backend.getQualities();
    };

    self.getChannel = function() {
        const { streamMetadata } = store.getState();
        return streamMetadata.channelName;
    };

    self.getVideo = function() {
        const { streamMetadata } = store.getState();
        return streamMetadata.videoId;
    };

    self.getSessionInfo = function() {
        const { analytics, manifestInfo } = store.getState();
        return {
            // eslint-disable-next-line camelcase
            broadcastId: manifestInfo.broadcast_id,
            playSessionId: analytics.playSessionId,
        };
    };

    self.startCast = function() {
        chromecast.load();
    };

    self.stopCast = function() {
        chromecast.stop();
    };

    self.getFullscreen = function() {
        const { screenMode } = store.getState();
        return screenMode.isFullScreen;
    };

    self.setFullscreen = function(value) {
        store.dispatch(setFullScreen(value));
    };

    self.getFullscreenEnabled = function() {
        const { screenMode } = store.getState();
        return screenMode.canFullScreen;
    };

    self.getStatsEnabled = function() {
        return store.getState().stats.enabled;
    };

    self.setStatsEnabled = function() {
        // eslint-disable-next-line no-console
        console.warn('setStatsEnabled has been deprecated.');
    };

    self.getStats = function() {
        return store.getState().stats.videoStats;
    };

    /** alias for getStats to match the EmbedPlayer API */
    self.getPlaybackStats = self.getStats;

    self.getVideoInfo = function() {
        return backend.getVideoInfo();
    };

    self.getBackend = function() {
        return backend.getBackend();
    };

    self.submitVideoIssueReport = function(issueType) {
        let properties = {
            issue: issueType,
        };
        properties = extend(properties, self.getVideoInfo());
        store.dispatch(trackEvent('vid_issue_report', properties));
    };

    /**
     * Sets a new backend to be used as a playback container.
     *
     * @param {String} The backend type to attempt to utilize.
     */
    self.setBackend = function(value) {
        // since changing the backend requires us to load the page,
        // we will only change the backend if we have access to cache
        if (localStore.usesCache()) {
            // Save the backend in local storage for next load.
            localStore.set('backend', value);

            // Force reload to enable to backend selection
            store.getState().window.document.location.reload();
        }
    };

    self.getVersion = function() {
        return store.getState().backend.getVersion();
    };

    self.getViewerCount = function() {
        return store.getState().viewercount;
    };

    self.getCaption = function() {
        return backend.getCaption();
    };

    self.getEventEmitter = function() {
        return events;
    };

    // This is required because we want a play_session_id associated with mini
    // player events, some of which have to be triggered from web-client
    self.trackMiniPlayerAction = function(actionType, reason) {
        store.dispatch(trackEvent(SITE_MINIPLAYER_ACTION, {
            action: actionType,
            reason: reason,
        }));
    };

    // Define the getters/setters to be compatible with the <video> tag.
    // TODO Imagine a world with no IE8...
    function defineProperty(name, args) {
        try {
            // enumerable is required for extending/forwarding properties
            // eslint-disable-next-line no-param-reassign
            args = defaults(args, {
                enumerable: true,
            });
            Object.defineProperty(self, name, args);
        } catch (defPropException) {
            // IE8 only, have to use get/set methods.
        }
    }

    defineProperty('error', {
        get: self.getError,
    });

    defineProperty('src', {
        get: self.getSrc,
        set: self.setSrc,
    });

    defineProperty('networkState', {
        get: self.getNetworkState,
    });

    defineProperty('buffered', {
        get: self.getBuffered,
    });

    defineProperty('readyState', {
        get: self.getReadyState,
    });

    defineProperty('seeking', {
        get: self.getSeeking,
    });

    defineProperty('currentTime', {
        get: self.getCurrentTime,
        set: self.setCurrentTime,
    });

    defineProperty('duration', {
        get: self.getDuration,
    });

    defineProperty('paused', {
        get: self.getPaused,
    });

    defineProperty('playbackRate', {
        get: self.getPlaybackRate,
        set: self.setPlaybackRate,
    });

    defineProperty('played', {
        get: self.getPlayed,
    });

    defineProperty('ended', {
        get: self.getEnded,
    });

    defineProperty('autoplay', {
        get: self.getAutoplay,
    });

    defineProperty('loop', {
        set: self.setLoop,
    });

    defineProperty('controls', {
        get: self.getControls,
        set: self.setControls,
    });

    defineProperty('volume', {
        get: self.getVolume,
        set: self.setVolume,
    });

    defineProperty('muted', {
        get: self.getMuted,
        set: self.setMuted,
    });

    defineProperty('quality', {
        get: self.getQuality,
        set: self.setQuality,
    });

    defineProperty('qualities', {
        get: self.getQualities,
    });

    defineProperty('channel', {
        get: self.getChannel,
        set: self.setChannel,
    });

    defineProperty('video', {
        get: self.getVideo,
        set: self.setVideo,
    });

    defineProperty('stats', {
        get: self.getStats,
    });

    defineProperty('statsEnabled', {
        get: self.getStatsEnabled,
        set: self.setStatsEnabled,
    });

    defineProperty('fullscreen', {
        get: self.getFullscreen,
        set: self.setFullscreen,
    });

    defineProperty('fullscreenEnabled', {
        get: self.getFullscreenEnabled,
    });

    defineProperty('theatre', {
        get: self.getTheatre,
        set: self.setTheatre,
    });

    defineProperty('viewers', {
        get: self.getViewerCount,
    });

    init();
}
