import { subscribe } from './util/subscribe';
import { requestAds, AdContentTypes, setVodMidrollArchiveSetting } from './actions/ads';
import { MIDROLL } from './ads/ima-tags';
import { IntervalTimer } from './util/interval-timer';
import { getChannelAdProperties } from './api';

export const TIMER_AD = 'TIMER_AD';
export const VOD_MIDROLL_SETTING_BROADCASTER = 'broadcaster';
export const VOD_MIDROLL_SETTING_INSERTED = 'inserted';
export const ARCHIVE = 'archive';

export class VodMidrollManager {
    constructor(store) {
        this._store = store;
        this._unsubs = [];
        this._timer = null;
        this._initSubscribes();
    }

    _initSubscribes() {
        // eslint-disable-next-line max-len
        this._unsubs.push(subscribe(this._store, ['ads.currentMetadata.contentType'], this.onContentTypeChange.bind(this)));
        this._unsubs.push(subscribe(this._store, ['playback.paused',
            'playback.playing'], this.onPlaybackStateChange.bind(this)));
        this._unsubs.push(subscribe(this._store, ['streamMetadata'], this.onStreamMetadataChange.bind(this)));
        this.onStreamMetadataChange(this._store.getState());
    }

    onContentTypeChange({ ads }) {
        if (ads.currentMetadata.contentType === AdContentTypes.NONE) {
            const streamMetadata = this._store.getState().streamMetadata;
            if (this._timer && streamMetadata.type === ARCHIVE &&
                this._store.getState().ads.vodMidrollArchiveSetting === VOD_MIDROLL_SETTING_INSERTED) {
                this._timer.start();
            }
        } else if (this._timer) {
            this._timer.pause();
        }
    }

    onPlaybackStateChange({ playback }) {
        if (playback.playing) {
            const streamMetadata = this._store.getState().streamMetadata;
            const ads = this._store.getState().ads;
            if (this._timer && streamMetadata.type === ARCHIVE &&
                ads.currentMetadata.contentType === AdContentTypes.NONE &&
                ads.vodMidrollArchiveSetting === VOD_MIDROLL_SETTING_INSERTED) {
                this._timer.start();
            }
        }

        if (playback.paused && this._timer) {
            this._timer.pause();
        }
    }

    onStreamMetadataChange({ streamMetadata }) {
        const { channel, type } = streamMetadata;
        const { name } = channel;
        if (type === ARCHIVE) {
            this.createTimer(name);
            this.onPlaybackStateChange(this._store.getState());
        } else if (this._timer) {
            this._timer.destroy();
            this._timer = null;
        }
    }

    createTimer(channelName) {
        /* eslint-disable camelcase */
        getChannelAdProperties(channelName).then(channelAdProperties => {
            const { valid_responses } = channelAdProperties;
            const { vod_ads_enabled } = valid_responses;
            if (vod_ads_enabled === true) {
                const { vod_archive_midrolls } = valid_responses;
                this._store.dispatch(setVodMidrollArchiveSetting(vod_archive_midrolls));
                if (vod_archive_midrolls === VOD_MIDROLL_SETTING_INSERTED) {
                    const { vod_archive_midrolls_break_length, vod_archive_midrolls_frequency } = valid_responses;
                    if (this._timer) {
                        this._timer.destroy();
                    }
                    this._timer = new IntervalTimer(() => {
                        // eslint-disable-next-line max-len
                        this._store.dispatch(requestAds(MIDROLL, parseInt(vod_archive_midrolls_break_length, 10), false, 0, TIMER_AD));
                    }, parseInt(vod_archive_midrolls_frequency, 10) * 60 * 1000, window);
                }
            }
        });
        /* eslint-enable */
    }

    destroy() {
        this._unsubs.forEach(unsub => unsub());
        if (this._timer) {
            this._timer.destroy();
        }
    }
}
