import sinon from 'sinon';
import { DEFAULT_STATS } from 'state/stats';

const DEFAULT_QUALITY = {
    name: '',
    group: '',
    codecs: '',
    bitrate: 0,
    width: 0,
    height: 0,
    framerate: 0,
};

export default class MediaPlayer {
    constructor() {
        this.quality = DEFAULT_QUALITY;
        this.eventHandlers = {};
        this.stats = DEFAULT_STATS;
        this.videoInfo = {};
        this.manifestInfo = {};
        this.media = document.createElement('video');
        sinon.spy(this, 'addEventListener');
    }

    addEventListener(name, handler) {
        if (this.eventHandlers[name]) {
            this.eventHandlers[name].push(handler);
        } else {
            this.eventHandlers[name] = [handler];
        }
    }

    removeEventListener(name, removeHandler) {
        if (removeHandler) {
            this.eventHandlers[name] = this.eventHandlers[name].filter(handler => handler !== removeHandler);
        } else {
            this.eventHandlers[name] = [];
        }
    }

    getHTMLVideoElement() {
        return this.media;
    }

    // Used only for testing. Not available in player-core
    emit(name, data) {
        const handlers = this.eventHandlers[name] || [];
        handlers.forEach(handler => handler(data));
    }

    load() {}

    _setPlaybackStatistics(stats) {
        this.stats = stats;
    }

    _setVideoInfo(videoInfo) {
        this.videoInfo = videoInfo;
    }

    _setManifestInfo(manifestInfo) {
        this.manifestInfo = manifestInfo;
    }

    _setMedia(media) {
        this.media = media;
    }

    getAutoSwitchQuality() {
        return true;
    }

    setQuality(quality) {
        this.quality = quality;
    }

    getQuality() {
        return this.quality;
    }

    getBufferDuration() {
        return this.stats.bufferSize;
    }

    getBuffered() {
        return {
            start: 0,
            end: 0,
        };
    }

    getQualities() {
        return [
            {
                group: 'chunked',
                name: 'Source',
                bitrate: 5120,
            },
            {
                group: 'high',
                name: 'High',
                bitrate: 4096,
            },
            {
                group: 'medium',
                name: 'Medium',
                bitrate: 2048,
            },
            {
                group: 'low',
                name: 'Low',
                bitrate: 1024,
            },
        ];
    }

    getManifestInfo() {
        return this.manifestInfo;
    }

    getPlaybackRate() {
        return this.stats.playbackRate;
    }

    getVideoFrameRate() {
        return this.stats.fps;
    }

    getVideoBitRate() {
        return this.stats.bitrate;
    }

    getDuration() {
        return 10;
    }

    getVolume() {
        return this.media.volume;
    }

    getDroppedFrames() {
        return this.stats.skippedFrames;
    }

    getTranscoderLatency() {
        return this.stats.hlsLatencyEncoder;
    }

    getBroadcasterLatency() {
        return this.stats.hlsLatencyBroadcaster;
    }

    getVideoWidth() {
        return this.stats.videoWidth;
    }

    getVideoHeight() {
        return this.stats.videoHeight;
    }

    getDisplayWidth() {
        return this.stats.displayWidth;
    }

    getDisplayHeight() {
        return this.stats.displayHeight;
    }

    getPosition() {
        return 0;
    }

    delete() {}

    getVersion() {
        return 'v1.0.0';
    }

    isMuted() {
        return this.media.muted;
    }

    play() {
        return this.media.play();
    }

    pause() {
        return this.media.pause();
    }

    setVolume(volume) {
        this.media.volume = volume;
    }

    setMuted(mute) {
        this.media.muted = mute;
    }

    getPlayerState() {
        return MediaPlayer.PlayerState.IDLE;
    }

    seekTo(time) {
        this.media.currentTime = time;
    }

    isSeeking() {
        return false;
    }

    getABSStats() {
        return {};
    }
}

MediaPlayer.BACKEND_MEDIA_PLAYER = 'mediaplayer';

MediaPlayer.PlayerEvent = {
    INITIALIZED: 'initialized',
    QUALITY_CHANGED: 'qualityChanged',
    DURATION_CHANGED: 'durationChanged',
    PLAYBACK_RATE_CHANGED: 'playbackRateChanged',
    REBUFFERING: 'rebuffering',
    ERROR: 'mediaplayererror',
    TIME_UPDATE: 'timeupdate',
    BUFFER_UPDATE: 'bufferupdate',
    SEEK_COMPLETED: 'seekCompleted',
    TRACKING: 'tracking',
};

MediaPlayer.MetadataEvent = {
    ID3: 'id3',
    CAPTION: 'caption',
    SPLICE_OUT: 'spliceOut',
    SPLICE_IN: 'spliceIn',
};

MediaPlayer.PlayerState = {
    IDLE: 'Idle',
    READY: 'Ready',
    BUFFERING: 'Buffering',
    PLAYING: 'Playing',
    ENDED: 'Ended',
};

MediaPlayer.ErrorType = {
    GENERIC: 'Error',

    /** Method or feature not supported */
    NOT_SUPPORTED: 'ErrorNotSupported',

    /** No source present for the operation */
    NO_SOURCE: 'ErrorNoSource',

    /** Data or input is invalid for the operation */
    INVALID_DATA: 'ErrorInvalidData',

    /** Class or object is an invalid state */
    INVALID_STATE: 'ErrorInvalidState',

    /** Method parameter is invalid */
    INVALID_PARAMETER: 'ErrorInvalidParameter',

    /** Method or operation timed out */
    TIMEOUT: 'ErrorTimeout',

    /** Output is invalid for the operation */
    INVALID_OUTPUT: 'ErrorInvalidOutput',

    /** An unexpected internal error has occurred. */
    INTERNAL: 'ErrorInternal',

    /** Unspecified Network error */
    NETWORK: 'ErrorNetwork',

    /** Error indicating a network I/O failure */
    NETWORK_IO: 'ErrorNetworkIO',

    /** Error indicating a network resource is not authorized */
    AUTHORIZATION: 'ErrorAuthorization',

    /** Error indicating a network resource is not available */
    NOT_AVAILABLE: 'ErrorNotAvailable',
};

MediaPlayer.ErrorSource = {
    UNKNOWN: 'Unspecified',
    FILE: 'File',
    PLAYLIST: 'Playlist',
    SEGMENT: 'Segment',
    SOURCE: 'Source',
    DECODER: 'Decode',
    RENDERER: 'Render',
};
