import sinon from 'sinon';
import { localStore } from 'tests/fakes/local-store.fake';
import * as CaptionsActions from 'actions/captions';
import { presetMap, fontSizeMap } from 'captions/captions-style-map';
import { unitTest } from 'tests/utils/module';

unitTest('actions | captions', function() {
    QUnit.test('showCCModal is formatted correctly', function(assert) {
        const action = CaptionsActions.showCCModal(true);

        const expectedAction = {
            type: CaptionsActions.ACTION_SHOW_CC_MODAL,
            showCCModal: true,
        };

        assert.deepEqual(action, expectedAction);
    });

    QUnit.module('initializeCaptionsSettings', function() {
        QUnit.test('returns Initialize Captions Settings action w/ defaults', function(assert) {
            const action = CaptionsActions.initializeCaptionsSettings();

            assert.equal(action.type, CaptionsActions.ACTION_INITIALIZE_CAPTIONS_SETTINGS);
            assert.equal(action.captions.enabled, false);
            assert.equal(action.captions.preset, 'white-on-black');
            assert.equal(action.captions.style, presetMap['white-on-black']);
        });

        QUnit.test('returns an Initialize Captions Settings action', function(assert) {
            localStore.set(CaptionsActions.KEY_CAPTIONS_PRESET, 'myFakePreset');
            localStore.set(CaptionsActions.KEY_CAPTIONS_ENABLED, true);

            const action = CaptionsActions.initializeCaptionsSettings();

            assert.equal(action.type, CaptionsActions.ACTION_INITIALIZE_CAPTIONS_SETTINGS);
            assert.equal(action.captions.enabled, true);
            assert.equal(action.captions.preset, 'myFakePreset');
        });

        QUnit.test('captions are forced enabled if action is passed true', function(assert) {
            localStore.set(CaptionsActions.KEY_CAPTIONS_ENABLED, false);
            const action = CaptionsActions.initializeCaptionsSettings(true);

            assert.equal(action.type, CaptionsActions.ACTION_INITIALIZE_CAPTIONS_SETTINGS);
            assert.equal(action.captions.enabled, true);
        });
    });

    QUnit.module('setCaptionsData', function() {
        QUnit.test('returns a Set Captions Data action', function(assert) {
            const data = { my: 'data' };
            const action = CaptionsActions.setCaptionsData(data);

            assert.equal(action.type, CaptionsActions.ACTION_SET_CAPTIONS_DATA);
            assert.deepEqual(action.captions.data, data);
        });
    });

    QUnit.module('setCaptionsEnabled', function(hooks) {
        hooks.beforeEach(function() {
            this.fakeState = () => {
                return {
                    captions: {
                        style: {},
                    },
                };
            };

            this.dispatchSpy = sinon.stub.returnsArg(0);
        });

        QUnit.test('when called with "true"', function(assert) {
            const action = CaptionsActions.setCaptionsEnabled(true)(this.dispatchSpy, this.fakeState);

            assert.equal(action.type, CaptionsActions.ACTION_TOGGLE_CAPTIONS);
            assert.equal(action.captions.enabled, true, 'creates an action with "enabled: true"');
            assert.equal(action.captions.style.display, 'block', 'sets the container\'s display to block');
            assert.equal(localStore.get(CaptionsActions.KEY_CAPTIONS_ENABLED), action.captions.enabled);
        });

        QUnit.test('when called with "false"', function(assert) {
            const action = CaptionsActions.setCaptionsEnabled(false)(this.dispatchSpy, this.fakeState);

            assert.equal(action.captions.enabled, false, 'creates an action with "enabled: false"');
            assert.equal(action.captions.style.display, 'none', 'sets the container\'s display to none');
        });
    });

    QUnit.module('setCaptionsPreset', function(hooks) {
        hooks.beforeEach(function() {
            this.fakeState = () => {
                return {
                    captions: {
                        style: {
                            color: 'black',
                            font: 'arial',
                            fontSize: 20,
                        },
                    },
                };
            };

            this.dispatchSpy = sinon.spy();
        });
        QUnit.test('returns Set Captions Preset action, writes to local store', function(assert) {
            const actionFn = CaptionsActions.setCaptionsPreset('custom', { color: 'white' });
            actionFn(this.dispatchSpy, this.fakeState);

            assert.deepEqual(this.dispatchSpy.firstCall.args[0], {
                type: CaptionsActions.ACTION_SET_CAPTIONS_PRESET,
                captions: {
                    preset: 'custom',
                    style: {
                        color: 'white',
                        font: 'arial',
                        fontSize: 20,
                    },
                },
            });
            assert.equal(localStore.get(CaptionsActions.KEY_CAPTIONS_PRESET), 'custom');
            assert.deepEqual(localStore.get(CaptionsActions.KEY_CAPTIONS_STYLE), {
                color: 'white',
                font: 'arial',
                fontSize: 20,
            });
        });

        QUnit.module('validate', function() {
            QUnit.test('constrains font size to be within valid bounds', function(assert) {
                let actionFn = CaptionsActions.setCaptionsPreset('custom', { fontSize: 50 });
                actionFn(this.dispatchSpy, this.fakeState);

                assert.deepEqual(this.dispatchSpy.firstCall.args[0], {
                    type: CaptionsActions.ACTION_SET_CAPTIONS_PRESET,
                    captions: {
                        preset: 'custom',
                        style: {
                            color: 'black',
                            font: 'arial',
                            fontSize: fontSizeMap.max,
                        },
                    },
                });

                actionFn = CaptionsActions.setCaptionsPreset('custom', { fontSize: 5 });
                actionFn(this.dispatchSpy, this.fakeState);

                assert.deepEqual(this.dispatchSpy.secondCall.args[0], {
                    type: CaptionsActions.ACTION_SET_CAPTIONS_PRESET,
                    captions: {
                        preset: 'custom',
                        style: {
                            color: 'black',
                            font: 'arial',
                            fontSize: fontSizeMap.min,
                        },
                    },
                });
            });
        });
    });
});
