import * as UniqueId from 'util/unique-id';
import * as CollectionActions from 'actions/collection';
import { TEST_COLLECTION, TEST_NORMALIZED_COLLECTION, TEST_COLLECTION_METADATA,
         TEST_NORMALIZED_FEATURED_COLLECTION_METADATA } from 'tests/fixtures/collection';
import { sessionStore } from 'tests/fakes/local-store.fake';
import sinon from 'sinon';
import isFunction from 'lodash/isFunction';

QUnit.module('actions | collection', function(hooks) {
    hooks.beforeEach(function() {
        this.collection = TEST_COLLECTION;
    });

    QUnit.test('requestCollection creates action with type `request collection`', function(assert) {
        const collectionId = 'a collection id';
        const videoId = 'a video id';
        const timestamp = 1234;
        const preferVideo = true;
        const expectedAction = {
            type: CollectionActions.ACTION_REQUEST_COLLECTION,
            request: {
                collectionId,
                videoId,
                timestamp,
                preferVideo,
            },
        };

        const action = CollectionActions.requestCollection(collectionId, videoId, timestamp, preferVideo);
        assert.deepEqual(action, expectedAction);
    });

    // eslint-disable-next-line max-len
    QUnit.test('requestCollection creates correct action and defaults videoId and timestamp to empty string', function(assert) {
        const collectionId = 'a collection id';
        const expectedAction = {
            type: CollectionActions.ACTION_REQUEST_COLLECTION,
            request: {
                collectionId,
                videoId: '',
                timestamp: '',
                preferVideo: false,
            },
        };

        const action = CollectionActions.requestCollection(collectionId);
        assert.deepEqual(action, expectedAction);
    });

    QUnit.test('featuredCollectionFetched normalizes featured collection data', function(assert) {
        const action = CollectionActions.featuredCollectionFetched(TEST_COLLECTION_METADATA);

        const expectedAction = {
            type: CollectionActions.ACTION_FETCHED_FEATURED_COLLECTION,
            featuredCollection: TEST_NORMALIZED_FEATURED_COLLECTION_METADATA,
        };

        assert.deepEqual(action, expectedAction);
    });

    QUnit.module('setCollection', function(hooks) {
        hooks.beforeEach(function() {
            sessionStore.clear();
        });

        hooks.afterEach(function() {
            sessionStore.clear();
        });

        QUnit.test('makes a dispatch to set collection session data and collection metadata', function(assert) {
            const collectionSessionData = sessionStore.get('collection-session', {});
            assert.deepEqual(collectionSessionData, {}, 'session store starts empty');

            const thunk = CollectionActions.setCollection(this.collection);
            const dispatchSpy = sinon.spy();
            thunk(dispatchSpy);

            assert.equal(dispatchSpy.callCount, 2, '2 dispatches are made');

            // First dispatch is for session store data.
            const setCollectionSessionDataThunk = dispatchSpy.firstCall.args[0];
            setCollectionSessionDataThunk();
            assert.ok(sessionStore.has('collection-session'), 'collection-session exists in session store');

            // Second dispatch is for redux store data.
            const action = dispatchSpy.secondCall.args[0];

            const expectedAction = {
                type: CollectionActions.ACTION_SET_COLLECTION,
                collection: TEST_NORMALIZED_COLLECTION,
            };
            assert.deepEqual(action, expectedAction, 'dispatches action to set collection metadata');
        });
    });

    QUnit.module('setCollectionSessionData', function(hooks) {
        hooks.beforeEach(function() {
            sessionStore.clear();
            this.now = Date.now();
            sinon.stub(Date, 'now').returns(this.now);
            sinon.stub(UniqueId, 'generate').returns('a new collection session id');
        });

        hooks.afterEach(function() {
            sessionStore.clear();
            Date.now.restore();
            UniqueId.generate.restore();
        });

        QUnit.test('when there is no previous collection data, create new session data', function(assert) {
            assert.deepEqual(
                sessionStore.get('collection-session', {}),
                {},
                'session store initially has no collection session data'
            );

            const thunk = CollectionActions.setCollectionSessionData(this.collection._id);
            assert.ok(isFunction(thunk), 'returns a thunk');
            thunk();

            const expectedSessionData = {
                'session-id': 'a new collection session id',
                'collection-id': this.collection._id,
                timestamp: this.now,
            };
            assert.deepEqual(
                sessionStore.get('collection-session', {}),
                expectedSessionData,
                'session store has collection data and a session id'
            );
        });

        QUnit.module('when there is previous collection session data', function(hooks) {
            hooks.beforeEach(function() {
                this.collectionSessionData = {
                    'session-id': 'a collection session id',
                    'collection-id': this.collection._id,
                    timestamp: this.now,
                };
                sessionStore.set('collection-session', this.collectionSessionData);
            });

            QUnit.test('keep the collection session id for tab refreshes (same collection)', function(assert) {
                sessionStore.set('collection-session', this.collectionSessionData);
                assert.deepEqual(
                    sessionStore.get('collection-session', {}),
                    this.collectionSessionData,
                    'session store starts with collection session data'
                );

                Date.now.onCall(0).returns(this.now + 100);
                const thunk = CollectionActions.setCollectionSessionData(this.collection._id);
                assert.ok(isFunction(thunk), 'returns a thunk');
                thunk();

                assert.deepEqual(
                    sessionStore.get('collection-session', {}),
                    Object.assign({}, this.collectionSessionData, {
                        timestamp: this.now + 100,
                    }),
                    'session store has the same session id with a new timestamp'
                );
            });

            QUnit.test('generate a new collection session id for stale sessions (same collection)', function(assert) {
                sessionStore.set('collection-session', this.collectionSessionData);
                assert.deepEqual(
                    sessionStore.get('collection-session', {}),
                    this.collectionSessionData,
                    'session store starts with collection session data'
                );

                Date.now.onCall(0).returns(this.now + CollectionActions.COLLECTION_SESSION_TIME_THRESHOLD);
                const thunk = CollectionActions.setCollectionSessionData(this.collection._id);
                assert.ok(isFunction(thunk), 'returns a thunk');
                thunk();

                const expectedSessionData = {
                    'session-id': 'a new collection session id',
                    'collection-id': this.collection._id,
                    timestamp: this.now + CollectionActions.COLLECTION_SESSION_TIME_THRESHOLD,
                };

                assert.deepEqual(
                    sessionStore.get('collection-session', {}),
                    expectedSessionData,
                    'session store has a new session id with a new timestamp'
                );
            });

            QUnit.test('generate a new collection session id for new sessions (diff collection)', function(assert) {
                sessionStore.set('collection-session', this.collectionSessionData);
                assert.deepEqual(
                    sessionStore.get('collection-session', {}),
                    this.collectionSessionData,
                    'session store starts with collection session data'
                );

                const thunk = CollectionActions.setCollectionSessionData('a different collection id');
                assert.ok(isFunction(thunk), 'returns a thunk');
                thunk();

                const expectedSessionData = {
                    'session-id': 'a new collection session id',
                    'collection-id': 'a different collection id',
                    timestamp: this.now,
                };

                assert.deepEqual(
                    sessionStore.get('collection-session', {}),
                    expectedSessionData,
                    'session store has a new collection, timestamp, and session id'
                );
            });
        });
    });

    QUnit.test('clearCollection clears collection session data and collection metadata', function(assert) {
        sessionStore.set('collection-session', { 'session-id': 'fake session id' });
        assert.ok(sessionStore.has('collection-session'), 'session store starts with data');

        const thunk = CollectionActions.clearCollection();
        const dispatchSpy = sinon.spy();
        thunk(dispatchSpy);

        assert.equal(dispatchSpy.callCount, 2, '2 dispatches are made');

        // First dispatch is for session store data.
        const clearCollectionSessionDataThunk = dispatchSpy.firstCall.args[0];
        clearCollectionSessionDataThunk();
        assert.notOk(sessionStore.has('collection-session'), 'collection-session is cleared from session store');

        // Second dispatch is for redux store data.
        const action = dispatchSpy.secondCall.args[0];

        const expectedAction = { type: CollectionActions.ACTION_CLEAR_COLLECTION };
        assert.deepEqual(action, expectedAction, 'dispatches action to set collection metadata');
    });

    QUnit.test('clearCollectionSessionData removes session data from session storage', function(assert) {
        sessionStore.set('collection-session', { 'session-id': 'abc123' });
        assert.deepEqual(
            sessionStore.get('collection-session', {}),
            { 'session-id': 'abc123' },
            'session store starts with collection session data'
        );

        const thunk = CollectionActions.clearCollectionSessionData();
        assert.ok(isFunction(thunk), 'returns a thunk');
        thunk();

        assert.notOk(sessionStore.has('collection-session'), 'session store has no collection session');
    });

    QUnit.module('updateCollectionSessionTimestamp', function(hooks) {
        hooks.beforeEach(function() {
            this.now = Date.now();
            sinon.stub(Date, 'now').returns(this.now);
            sinon.stub(UniqueId, 'generate').returns('a new collection session id');
        });

        hooks.afterEach(function() {
            sessionStore.clear();
            Date.now.restore();
            UniqueId.generate.restore();
        });

        QUnit.test('updateCollectionSessionTimestamp updates the timestamp in session store', function(assert) {
            sessionStore.set('collection-session', { timestamp: 1 });
            let collectionSessionData = sessionStore.get('collection-session', {});
            assert.equal(collectionSessionData.timestamp, 1, 'session timestamp starts as 1');

            const thunk = CollectionActions.updateCollectionSessionTimestamp();
            thunk();

            collectionSessionData = sessionStore.get('collection-session', {});
            assert.equal(collectionSessionData.timestamp, this.now, 'timestamp is updated with the current time');
        });
    });

    QUnit.test('openCollectionSidebar creates action with type `open collection`', function(assert) {
        const action = CollectionActions.openCollectionSidebar();

        const expectedAction = { type: CollectionActions.ACTION_OPEN_COLLECTION_SIDEBAR };

        assert.deepEqual(action, expectedAction);
    });

    QUnit.test('closeCollectionSidebar creates action with type `close collection`', function(assert) {
        const action = CollectionActions.closeCollectionSidebar();

        const expectedAction = { type: CollectionActions.ACTION_CLOSE_COLLECTION_SIDEBAR };

        assert.deepEqual(action, expectedAction);
    });

    QUnit.test('loadedLastCollectionItem creates action with type `loaded last collection item`', function(assert) {
        const action = CollectionActions.loadedLastCollectionItem();

        const expectedAction = { type: CollectionActions.ACTION_LOADED_LAST_COLLECTION_ITEM };

        assert.deepEqual(action, expectedAction);
    });

    QUnit.test('loadedCollectionItem creates action with type `loaded collection item`', function(assert) {
        const action = CollectionActions.loadedCollectionItem();

        const expectedAction = { type: CollectionActions.ACTION_LOADED_COLLECTION_ITEM };

        assert.deepEqual(action, expectedAction);
    });
});
