import { ExtensionCoordinator } from 'extension-coordinator';
import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { createExtension } from 'tests/fixtures/extensions';
import * as ExtensionsActions from 'actions/extensions';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { AdRollTypes } from 'actions/ads';
import { init as initStore } from 'state';

const TEST_CHANNEL_NAME = 'twitch';
const HOSTED_TEST_CHANNEL_NAME = 'not_twitch';
const TEST_CHANNEL_ID = 12345;
const HOSTED_TEST_CHANNEL_ID = 54321;

const ACTION_FETCHING_EXTENSIONS = {
    type: ExtensionsActions.ACTION_FETCHING_EXTENSIONS,
    channel: TEST_CHANNEL_NAME,
};

const ACTION_FETCHING_EXTENSIONS_HOSTED = {
    type: ExtensionsActions.ACTION_FETCHING_EXTENSIONS,
    channel: HOSTED_TEST_CHANNEL_NAME,
};

unitTest('actions | extensions', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.oauthToken = this.api.setLoggedIn(true);
        this.stream = new LiveTwitchContentStream(
            TEST_CHANNEL_NAME,
            this.oauthToken,
            {}
        );
        this.hostedStream = new LiveTwitchContentStream(
            HOSTED_TEST_CHANNEL_NAME,
            this.oauthToken,
            {}
        );
    });

    QUnit.module('fetchExtensions', function(hooks) {
        hooks.beforeEach(function() {
            this.userInfo = this.api.expectUserInfo();
        });

        QUnit.module('when the Extensions experiment is disabled', function(hooks) {
            hooks.beforeEach(function() {
                this.getState = () => ({
                    extensions: {
                        channel: undefined,
                        extensions: [],
                        loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
                    },
                    stream: this.stream,
                });
            });

            QUnit.test('the extensions request is resolved as loaded with no extensions', function(assert) {
                const dispatch = sinon.spy();
                const action = ExtensionsActions.fetchExtensions(TEST_CHANNEL_NAME);

                return action(dispatch, this.getState).
                    then(() => {
                        assert.ok(dispatch.calledWith(sinon.match({
                            type: ExtensionsActions.ACTION_SET_EXTENSIONS,
                            channel: TEST_CHANNEL_NAME,
                            extensions: [],
                        })));
                    });
            });
        });

        QUnit.module('when the user is watching a stream with overlay extensions installed', function(hooks) {
            let respExtensions = null;
            hooks.beforeEach(function() {
                this.api.expectChannelInfo(TEST_CHANNEL_NAME, { _id: TEST_CHANNEL_ID });
                respExtensions = this.api.expectOverlayExtensionsForChannel(TEST_CHANNEL_ID, 1);

                const getStateInfo = {
                    extensions: {
                        channel: undefined,
                        extensions: [],
                        loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
                    },
                    playback: {
                        isLoading: true,
                        paused: false,
                        contentShowing: true,
                    },
                    stream: this.stream,
                    ads: { currentMetadata: { rollType: AdRollTypes.NONE } },
                };

                this.store = {
                    getState: () => (getStateInfo),
                };
            });

            QUnit.test('fetches extensions for the current stream', function(assert) {
                const dispatch = sinon.spy();
                const action = ExtensionsActions.fetchExtensions(TEST_CHANNEL_NAME);
                const extensions = respExtensions.installed_extensions.map(install => {
                    const {
                        id,
                        name,
                        sku,
                        summary,
                        vendor_code: vendorCode,
                        version,
                        state,
                        viewer_url: viewerUrl,
                        request_identity_link: supportsIdentityLinking,
                        anchor,
                    } = install.extension;
                    return {
                        id,
                        name,
                        sku,
                        summary,
                        vendorCode,
                        version,
                        state,
                        viewerUrl,
                        supportsIdentityLinking,
                        lastUserIdentityLinkState: false,
                        anchor,
                        token: {
                            permissionsState: 'none',
                            role: 'viewer',
                            token: respExtensions.tokens.filter(t => t.extension_id === id)[0].token,
                        },
                    };
                });
                const expected = {
                    type: ExtensionsActions.ACTION_SET_EXTENSIONS,
                    extensions,
                    channel: TEST_CHANNEL_NAME,
                };

                return action(dispatch, this.store.getState).then(() => {
                    const { firstCall, secondCall } = dispatch;
                    assert.ok(firstCall, 'first call to dispatch was made');
                    assert.deepEqual(firstCall.args[0], ACTION_FETCHING_EXTENSIONS,
                        'dispatched a fetch extensions action');
                    assert.ok(secondCall, 'second call to dispatch was made');
                    assert.deepEqual(secondCall.args[0], expected,
                        'dispatched a set action with the expected array of extensions');
                });
            });
        });

        QUnit.module('when the user is watching a stream with no extensions installed', function(hooks) {
            hooks.beforeEach(function() {
                this.api.expectChannelInfo(TEST_CHANNEL_NAME, { _id: TEST_CHANNEL_ID });
                /* eslint-disable camelcase */
                const extension = createExtension();
                extension.anchor = 'panel';
                this.extensions = this.api.expectOverlayExtensionsForChannel(
                    TEST_CHANNEL_ID,
                    Object.freeze([extension])
                );
                /* eslint-enable camelcase */

                const getStateInfo = {
                    extensions: {
                        channel: undefined,
                        extensions: [],
                        loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
                    },
                    stream: this.stream,
                };

                this.store = {
                    getState: () => (getStateInfo),
                };
            });

            QUnit.test('fetches extensions for the current stream', function(assert) {
                const dispatch = sinon.spy();
                const action = ExtensionsActions.fetchExtensions(TEST_CHANNEL_NAME);
                const expected = {
                    type: ExtensionsActions.ACTION_SET_EXTENSIONS,
                    extensions: [],
                    channel: TEST_CHANNEL_NAME,
                };

                return action(dispatch, this.store.getState).then(() => {
                    const { firstCall, secondCall } = dispatch;
                    assert.ok(firstCall, 'first call to dispatch was made');
                    assert.deepEqual(firstCall.args[0], ACTION_FETCHING_EXTENSIONS,
                        'dispatched a fetch extension action');
                    assert.ok(secondCall, 'second call to dispatch was made');
                    assert.deepEqual(secondCall.args[0], expected,
                        'dispatched a set extensions action with an empty array');
                });
            });
        });

        QUnit.module('when the user switches between streams with activated extensions', function(hooks) {
            hooks.beforeEach(function() {
                this.state = {
                    extensions: {
                        channel: undefined,
                        extensions: [],
                        loadingState: ExtensionsActions.EXTENSIONS_NOT_LOADED,
                    },
                    playback: {
                        isLoading: true,
                        paused: false,
                        contentShowing: true,
                    },
                    stream: this.stream,
                    ads: { currentMetadata: { rollType: AdRollTypes.NONE } },
                };

                this.store = {
                    getState: () => (this.state),
                };
            });

            QUnit.test('fetches extensions for the correct stream', function(assert) {
                this.api.expectChannelInfo(TEST_CHANNEL_NAME, { _id: TEST_CHANNEL_ID });
                this.api.expectChannelInfo(HOSTED_TEST_CHANNEL_NAME, { _id: HOSTED_TEST_CHANNEL_ID });
                this.state.stream = this.hostedStream;

                this.api.expectOverlayExtensionsForChannel(TEST_CHANNEL_ID, 1, 250);
                const secondResponse = this.api.expectOverlayExtensionsForChannel(HOSTED_TEST_CHANNEL_ID, 1);

                const dispatch = sinon.spy();
                const firstAction = ExtensionsActions.fetchExtensions(TEST_CHANNEL_NAME);
                const firstPromise = firstAction(dispatch, this.store.getState);

                const secondAction = ExtensionsActions.fetchExtensions(HOSTED_TEST_CHANNEL_NAME);
                const secondPromise = secondAction(dispatch, this.store.getState);
                const secondExpected = {
                    type: ExtensionsActions.ACTION_SET_EXTENSIONS,
                    extensions: secondResponse.installed_extensions.map(install => {
                        const { id, name, sku, summary, version, state, viewer_url: viewerUrl } = install.extension;
                        return {
                            id,
                            name,
                            sku,
                            summary,
                            vendorCode: install.extension.vendor_code,
                            version,
                            state,
                            viewerUrl,
                            anchor: install.extension.anchor,
                            supportsIdentityLinking: install.extension.request_identity_link,
                            lastUserIdentityLinkState: false,
                            token: {
                                permissionsState: 'none',
                                role: 'viewer',
                                token: secondResponse.tokens.filter(t => t.extension_id === id)[0].token,
                            },
                        };
                    }),
                    channel: HOSTED_TEST_CHANNEL_NAME,
                };

                return Promise.all([firstPromise, secondPromise]).then(() => {
                    const { firstCall, secondCall, thirdCall } = dispatch;

                    assert.ok(firstCall, 'first call to dispatch was made');
                    assert.deepEqual(firstCall.args[0], ACTION_FETCHING_EXTENSIONS,
                        'dispatched a fetch extensions action');
                    assert.ok(secondCall, 'second call to dispatch was made');
                    assert.deepEqual(secondCall.args[0], ACTION_FETCHING_EXTENSIONS_HOSTED,
                        'dispatched a fetch extensions action');

                    assert.ok(thirdCall, 'third call to dispatch was made');
                    assert.deepEqual(thirdCall.args[0], secondExpected,
                        'dispatched a set action with the expected array of extensions');

                    assert.equal(dispatch.callCount, 3, 'no fourth call to dispatch was made');
                });
            });
        });
    });

    QUnit.module('subscribeToExtensionControl', function(hooks) {
        const RANDOM_NUMBER = 0.50;
        let triggerShouldFetchExtensions;
        let triggerShouldDestroy;

        hooks.beforeEach(function() {
            sinon.stub(Math, 'random').returns(RANDOM_NUMBER);

            this.channelName = `channel_${QUnit.config.current.testId}`;
            this.channelInfo = this.api.expectChannelInfo(this.channelName);

            this.getStateInfo = {
                extensions: {
                    channel: this.channelName,
                    activeOverlayExtensions: [],
                    loadingState: ExtensionsActions.EXTENSIONS_LOADED,
                },
                playback: {
                    isLoading: false,
                    paused: false,
                    contentShowing: true,
                },
                stream: this.stream,
                ads: { currentMetadata: { rollType: AdRollTypes.NONE } },
                viewercount: 40000,
            };

            this.store = {
                getState: () => (this.getStateInfo),
            };

            sinon.stub(
                ExtensionCoordinator.ExtensionService,
                'subscribeToExtensionControl',
                (channelId, controlHandlers) => {
                    triggerShouldFetchExtensions = controlHandlers.onShouldFetchExtensions;
                    triggerShouldDestroy = controlHandlers.onDestroyExtension;
                }
            );
        });

        hooks.afterEach(function() {
            Math.random.restore();
            ExtensionCoordinator.ExtensionService.subscribeToExtensionControl.restore();
        });

        QUnit.module('applies proper handlers to ExtensionService.subscribeToExtensionControl()', function() {
            QUnit.module('onShouldFetchExtensions', function(hooks) {
                let applyJitterCallback;
                let jitteredFunction;
                let jitterTimeout;
                let setTimeout;

                hooks.beforeEach(function() {
                    setTimeout = sinon.spy(func => {
                        jitteredFunction = func;
                    });
                    applyJitterCallback = () => jitteredFunction();
                    jitterTimeout = this.getStateInfo.viewercount /
                                          ExtensionsActions.EMS_REQUEST_PER_SECOND_LIMIT * 1000 *
                                          RANDOM_NUMBER;

                    this.getStateInfo.window = {
                        setTimeout,
                    };
                });

                QUnit.test('should early out if loadingState === EXTENSIONS_LOADING', function(assert) {
                    const action = ExtensionsActions.subscribeToExtensionControl(this.channelName);
                    const dispatch = sinon.spy();

                    return action(dispatch, this.store.getState).then(() => {
                        triggerShouldFetchExtensions();

                        assert.ok(
                            setTimeout.calledWith(
                                sinon.match(sinon.match.func),
                                jitterTimeout
                            )
                        );

                        applyJitterCallback();

                        const [firstDispatch, secondDispatch] = dispatch.getCalls();
                        assert.ok(
                            firstDispatch.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_EXTENSION_ACTIVATED,
                                channel: this.channelName,
                            })),
                            'should dispatch extension activated action'
                        );

                        this.getStateInfo.extensions.loadingState = ExtensionsActions.EXTENSIONS_LOADING;
                        const [fetchExtensionsAction] = secondDispatch.args;
                        fetchExtensionsAction(dispatch, this.store.getState);
                        const { lastCall } = dispatch;

                        assert.ok(
                            lastCall.notCalledWith(sinon.match({
                                type: ExtensionsActions.ACTION_FETCHING_EXTENSIONS,
                                channel: this.channelName,
                            })),
                            'should not dispatch fetchingExtensions'
                        );
                    });
                });

                QUnit.test('should fetchExtensions when loadingState EXTENSIONS_LOADED', function(assert) {
                    const action = ExtensionsActions.subscribeToExtensionControl(this.channelName);
                    const dispatch = sinon.spy();

                    return action(dispatch, this.store.getState).then(() => {
                        triggerShouldFetchExtensions();

                        assert.ok(
                            setTimeout.calledWith(
                                sinon.match(sinon.match.func),
                                jitterTimeout
                            )
                        );

                        applyJitterCallback();

                        const [firstDispatch, secondDispatch] = dispatch.getCalls();
                        assert.ok(
                            firstDispatch.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_EXTENSION_ACTIVATED,
                                channel: this.channelName,
                            })),
                            'should dispatch extension activated action'
                        );

                        const [fetchExtensionsAction] = secondDispatch.args;
                        fetchExtensionsAction(dispatch, this.store.getState);
                        const { lastCall } = dispatch;

                        assert.ok(
                            lastCall.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_FETCHING_EXTENSIONS,
                                channel: this.channelName,
                            })),
                            'should dispatch fetchingExtensions'
                        );
                    });
                });

                QUnit.test('should fetchExtensions when loadingState EXTENSIONS_RELOADING', function(assert) {
                    const dispatch = sinon.spy();
                    const action = ExtensionsActions.subscribeToExtensionControl(this.channelName);

                    return action(dispatch, this.store.getState).then(() => {
                        triggerShouldFetchExtensions();

                        assert.ok(
                            setTimeout.calledWith(
                                sinon.match(sinon.match.func),
                                jitterTimeout
                            )
                        );

                        applyJitterCallback();

                        const [firstCall, secondCall] = dispatch.getCalls();
                        assert.ok(
                            firstCall.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_EXTENSION_ACTIVATED,
                                channel: this.channelName,
                            })),
                            'should dispatch extension activated action'
                        );

                        this.getStateInfo.extensions.loadingState = ExtensionsActions.EXTENSIONS_RELOADING;
                        const [fetchExtensionsAction] = secondCall.args;
                        fetchExtensionsAction(dispatch, this.store.getState);
                        const { lastCall } = dispatch;

                        assert.ok(
                            lastCall.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_FETCHING_EXTENSIONS,
                                channel: this.channelName,
                            })),
                            'should dispatch fetchingExtensions'
                        );
                    });
                });
            });

            // eslint-disable-next-line max-len
            QUnit.module('onDestroyExtension should dispatch setExtensions with the new list of extensions', function() {
                QUnit.test('it should remove the extension if it is in the current list', function(assert) {
                    const existingExtension = createExtension();
                    this.getStateInfo.extensions.activeOverlayExtensions.push(existingExtension);

                    const dispatch = sinon.spy();
                    const action = ExtensionsActions.subscribeToExtensionControl(this.channelName);

                    return action(dispatch, this.store.getState).then(() => {
                        triggerShouldDestroy(existingExtension.id);
                        const [firstCall] = dispatch.getCalls();
                        assert.ok(
                            firstCall.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_SET_EXTENSIONS,
                                extensions: [],
                            })),
                            'should dispatch extension activated action'
                        );
                    });
                });

                QUnit.test('it should not remove any extennsions if it is not in the current list', function(assert) {
                    const existingExtension = createExtension();
                    this.getStateInfo.extensions.activeOverlayExtensions.push(existingExtension);

                    const dispatch = sinon.spy();
                    const action = ExtensionsActions.subscribeToExtensionControl(this.channelName);

                    return action(dispatch, this.store.getState).then(() => {
                        triggerShouldDestroy('notinlist');
                        const [firstCall] = dispatch.getCalls();
                        assert.ok(
                            firstCall.calledWith(sinon.match({
                                type: ExtensionsActions.ACTION_SET_EXTENSIONS,
                                extensions: [existingExtension],
                            })),
                            'should dispatch extension activated action'
                        );
                    });
                });
            });
        });
    });

    QUnit.test('setExtensionModal', function(assert) {
        const modalRequest = {
            action: 'twitch-ext-random-modal-request',
            options: {},
            resultCallback() {},
            defaultResult: {},
        };

        const expectedAction = {
            type: ExtensionsActions.ACTION_SET_EXTENSION_MODAL,
            modalRequest,
        };

        assert.deepEqual(
            ExtensionsActions.setExtensionModal(modalRequest),
            expectedAction,
            'should create a set modal action with the given modal request'
        );
    });

    QUnit.test('closeExtensionModal', function(assert) {
        const expectedAction = {
            type: ExtensionsActions.ACTION_SET_EXTENSION_MODAL,
            modalRequest: null,
        };

        assert.deepEqual(
            ExtensionsActions.closeExtensionModal(),
            expectedAction,
            'should create a set modal action with a null modal request'
        );
    });
});
