import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';
import * as FollowActions from 'actions/follow';

function generateIds() {
    const userId = parseInt(Math.random() * 1000, 10);
    const channelId = parseInt(Math.random() * 1000, 10);
    return {
        userId,
        channelId,
    };
}

unitTest('actions | follow', function() {
    QUnit.test('showFollowNotification is formatted correctly', function(assert) {
        const action = FollowActions.showFollowNotification(true);

        const expectedAction = {
            type: FollowActions.ACTION_SHOW_FOLLOW_NOTIFICATION,
            show: true,
        };

        assert.deepEqual(action, expectedAction);
    });

    QUnit.test('promptLoginModal should be the right shape', function(assert) {
        const channelName = QUnit.config.current.testId;
        const expected = {
            type: FollowActions.ACTION_PROMPT_LOGIN_MODAL,
            channelName,
        };

        assert.deepEqual(FollowActions.promptLoginModal(channelName), expected);
    });

    QUnit.test('followInfoFetched should be the right shape', function(assert) {
        const followInfo = {
            following: true,
            notificationsEnabled: true,
        };
        const expected = {
            type: FollowActions.ACTION_FOLLOW_INFO_FETCHED,
            followInfo,
        };
        assert.deepEqual(FollowActions.followInfoFetched(followInfo), expected);
    });

    QUnit.test('logNotificationsEnabled logs successful notifications enable', function(assert) {
        const trackEvent = sinon.spy();

        FollowActions.logNotificationsEnabled({ trackEvent });

        assert.equal(trackEvent.callCount, 2);

        const { firstCall, secondCall } = trackEvent;

        assert.deepEqual(firstCall.args, [
            'follow',
            { src: 'player' },
        ], 'logs successful follow');

        assert.deepEqual(secondCall.args, [
            'notification_change',
            { notifications: true },
        ], 'logs succssful notifications enabled');
    });

    QUnit.test('logNotificationsDisabled logs successful notifications disable', function(assert) {
        const trackEvent = sinon.spy();

        FollowActions.logNotificationsDisabled({ trackEvent });

        assert.equal(trackEvent.callCount, 1);

        const { firstCall } = trackEvent;

        assert.deepEqual(firstCall.args, [
            'notification_change',
            { notifications: false },
        ], 'logs successful notifications disable');
    });

    QUnit.module('dispatchFollowSuccess', function() {
        QUnit.test('validates then dispatches data from follow api', function(assert) {
            const dispatch = sinon.spy();
            const followResponse = {
                channel: {
                    _id: 12345,
                },
                notifications: false,
            };

            FollowActions.dispatchFollowSuccess(dispatch, followResponse);

            assert.equal(dispatch.callCount, 1);
            assert.deepEqual(dispatch.firstCall.args[0], FollowActions.followInfoFetched({
                following: true,
                notificationsEnabled: followResponse.notifications,
            }));
        });

        QUnit.test('no-ops if invalid response found', function(assert) {
            const dispatch = sinon.spy();
            const followResponse = {
                channel: {
                    _id: 12345,
                },
            };

            FollowActions.dispatchFollowSuccess(dispatch, followResponse);

            assert.equal(dispatch.callCount, 0);
        });
    });

    QUnit.module('dispatchFollowFail', function() {
        QUnit.test('on 404, dispatches followInfoFetched with { following: false }', function(assert) {
            const dispatch = sinon.spy();
            const { channelId } = generateIds();

            const followResponse = {
                status: 404,
            };

            FollowActions.dispatchFollowFail(dispatch, followResponse, channelId);

            assert.equal(dispatch.callCount, 1);
            assert.deepEqual(dispatch.firstCall.args[0], FollowActions.followInfoFetched({
                following: false,
                notificationsEnabled: false,
            }));
        });

        QUnit.test('no-ops on non-404 error responses', function(assert) {
            const dispatch = sinon.spy();
            const { channelId } = generateIds();

            const followResponse = {
                status: 500,
            };

            FollowActions.dispatchFollowFail(dispatch, followResponse, channelId);

            assert.equal(dispatch.callCount, 0);
        });
    });

    function testFetchFollowInfo(following) {
        const msg = `dispatches followInfo if user ${following ? 'is' : 'is not'} following channel`;
        QUnit.test(msg, function(assert) {
            const { channelId, userId } = generateIds();
            const stream = {};

            this.api.setLoggedIn(true);
            this.api.expectFollowInfo({
                userId,
                channelId,
                following,
                notificationsEnabled: false,
            });

            const dispatchSpy = sinon.spy();
            const getState = () => ({ stream });
            const action = FollowActions.fetchFollowInfo(userId, channelId);

            return action(dispatchSpy, getState).
                then(() => {
                    assert.equal(dispatchSpy.callCount, 2);
                    assert.deepEqual(dispatchSpy.firstCall.args[0], {
                        type: FollowActions.ACTION_FOLLOW_INFO_FETCHED,
                        followInfo: {
                            following,
                            notificationsEnabled: false,
                        },
                    });
                    assert.deepEqual(dispatchSpy.secondCall.args[0], {
                        type: FollowActions.ACTION_SHOW_FOLLOW_NOTIFICATION,
                        show: true,
                    });
                });
        });

        // eslint-disable-next-line max-len
        QUnit.test('should not dispatch followInfo if streams change during the follow info request', function(assert) {
            const { channelId, userId } = generateIds();

            this.api.setLoggedIn(true);
            this.api.expectFollowInfo({
                userId,
                channelId,
                following,
                notificationsEnabled: false,
            });

            const dispatchSpy = sinon.spy();
            const getState = () => ({ stream: {} }); // generates new stream object each time

            const action = FollowActions.fetchFollowInfo(userId, channelId);

            return action(dispatchSpy, getState).
                then(() => {
                    assert.equal(dispatchSpy.callCount, 0);
                });
        });
    }

    testFetchFollowInfo(true);
    testFetchFollowInfo(false);

    // eslint-disable-next-line max-len
    QUnit.test('fetchFollowInfo dispatches showFollowNotification false if user is following & notifications enabled', function(assert) {
        const { channelId, userId } = generateIds();
        const stream = {};

        this.api.setLoggedIn(true);
        this.api.expectFollowInfo({
            userId,
            channelId,
            following: true,
            notificationsEnabled: true,
        });

        const dispatchSpy = sinon.spy();
        const getState = () => ({ stream });
        const action = FollowActions.fetchFollowInfo(userId, channelId);

        return action(dispatchSpy, getState).
            then(() => {
                assert.equal(dispatchSpy.callCount, 2);
                assert.deepEqual(dispatchSpy.secondCall.args[0], {
                    type: FollowActions.ACTION_SHOW_FOLLOW_NOTIFICATION,
                    show: false,
                });
            });
    });

    // eslint-disable-next-line max-len
    QUnit.test('fetchFollowInfo dispatches showFollowNotification true if user is following but no notifications enabled', function(assert) {
        const { channelId, userId } = generateIds();
        const stream = {};

        this.api.setLoggedIn(true);
        this.api.expectFollowInfo({
            userId,
            channelId,
            following: true,
            notificationsEnabled: false,
        });

        const dispatchSpy = sinon.spy();
        const getState = () => ({ stream });
        const action = FollowActions.fetchFollowInfo(userId, channelId);

        return action(dispatchSpy, getState).
            then(() => {
                assert.equal(dispatchSpy.callCount, 2);
                assert.deepEqual(dispatchSpy.secondCall.args[0], {
                    type: FollowActions.ACTION_SHOW_FOLLOW_NOTIFICATION,
                    show: true,
                });
            });
    });

    function testEnableDisableNotifications(notificationsEnabled) {
        const methodName = notificationsEnabled ? 'enableNotifications' : 'disableNotifications';

        QUnit.test(`${methodName} should update store when request has been processed`, function(assert) {
            const { channelId, userId } = generateIds();

            this.api.setLoggedIn(true);
            this.api.expectSetNotifications(notificationsEnabled, {
                channelId,
            });

            const dispatchSpy = sinon.spy();
            const analyticsTracker = {
                trackEvent: sinon.spy(),
            };

            const action = notificationsEnabled ?
                FollowActions.enableNotifications(userId, channelId) :
                FollowActions.disableNotifications(userId, channelId);

            return action(dispatchSpy, () => ({ analyticsTracker })).
                then(() => {
                    assert.deepEqual(dispatchSpy.firstCall.args[0], {
                        type: FollowActions.ACTION_FOLLOW_INFO_FETCHED,
                        followInfo: {
                            following: true,
                            notificationsEnabled,
                        },
                    });

                    const { firstCall, secondCall } = analyticsTracker.trackEvent;

                    if (notificationsEnabled) {
                        assert.deepEqual(firstCall.args, [
                            'follow',
                            { src: 'player' },
                        ], 'logs successful follow');

                        assert.deepEqual(secondCall.args, [
                            'notification_change',
                            { notifications: true },
                        ], 'logs successful notifications enabled');
                    } else {
                        assert.deepEqual(firstCall.args, [
                            'notification_change',
                            { notifications: false },
                        ], 'logs successful notifications disabled');
                    }
                });
        });
    }

    testEnableDisableNotifications(true);
    testEnableDisableNotifications(false);

    function testFollowUnfollowChannel(following) {
        const methodName = following ? 'followChannel' : 'unfollowChannel';

        QUnit.test(`${methodName} should update store when request has been processed`, function(assert) {
            const { channelId, userId } = generateIds();

            this.api.setLoggedIn(true);
            this.api.expectSetFollow(following, {
                channelId,
                notificationsEnabled: false,
            });

            const dispatchSpy = sinon.spy();

            const action = following ?
                FollowActions.followChannel(userId, channelId) :
                FollowActions.unfollowChannel(userId, channelId);

            return action(dispatchSpy).
                then(() => {
                    assert.deepEqual(dispatchSpy.firstCall.args[0], {
                        type: FollowActions.ACTION_FOLLOW_INFO_FETCHED,
                        followInfo: {
                            following,
                            notificationsEnabled: false,
                        },
                    });
                });
        });
    }

    testFollowUnfollowChannel(true);
    testFollowUnfollowChannel(false);
});
