import assign from 'lodash/assign';
import * as PlaybackActions from 'actions/playback';
import isFunction from 'lodash/isFunction';
import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import * as MediaEvents from 'backend/events/media-event';
import { localStore } from 'tests/fakes/local-store.fake';
import { init as initStore } from 'state';
import { setAccessTokenParams } from 'actions/access-token';
import { ACTION_TRANSITION_TO_REC_VOD, ACTION_TRANSITION_TO_COLLECTION } from 'actions/event-emitter';
import { setPlayerType } from 'actions/env';
import { PLAYER_SITE, PLAYER_POPOUT, PLAYER_EMBED, PLAYER_TWITCH_EVERYWHERE } from 'util/player-type';
import { setCollection, requestCollection, loadedCollectionItem,
         ACTION_CLEAR_COLLECTION } from 'actions/collection';
import { ACTION_SET_STREAM } from 'actions/stream';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { setWindow } from 'actions/window';
import { setFullScreen } from 'actions/screen-mode';
import { TRANSITION_TYPE_RECOMMENDATIONS } from 'state/playback';
import { TEST_COLLECTION } from 'tests/fixtures/collection';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { parse } from 'util/params';

unitTest('actions/playback', function() {
    QUnit.module('initializePlaybackSettings', function(hooks) {
        hooks.beforeEach(function() {
            this.playback = {
                autoplay: true,
                buffer: {
                    start: 0,
                    end: 0,
                },
                bufferEmpties: 0,
                currentTime: 0,
                duration: 0,
                ended: false,
                hasPlayed: false,
                isLoading: false,
                muted: false,
                paused: false,
                restrictedQualityError: false,
                volume: 0.5,
                transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS,
                contentShowing: false,
            };
            this.getState = () => {
                return {
                    playback: this.playback,
                };
            };
        });

        QUnit.test('defaults values are used when cache is empty', function(assert) {
            const dispatch = sinon.spy();
            const action = PlaybackActions.initializePlaybackSettings();

            assert.ok(isFunction(action), 'initializePlaybackSettings should return a function');
            action(dispatch, this.getState);

            assert.equal(dispatch.callCount, 1, 'dispatch should be called');
            assert.ok(dispatch.firstCall.calledWith({
                type: PlaybackActions.ACTION_INITIALIZE_PLAYBACK_SETTINGS,
                playback: {
                    muted: false,
                    volume: 0.5,
                },
            }), 'playback is being initialized with expected values');
        });

        QUnit.test('uses cached values when available', function(assert) {
            const muted = true;
            const volume = 0.2;

            localStore.set('muted', muted);
            localStore.set('volume', volume);

            const dispatch = sinon.spy();
            const action = PlaybackActions.initializePlaybackSettings();

            assert.ok(isFunction(action));
            action(dispatch, this.getState);

            assert.equal(dispatch.callCount, 1, 'dispatch should be called');
            assert.ok(dispatch.firstCall.calledWith({
                type: PlaybackActions.ACTION_INITIALIZE_PLAYBACK_SETTINGS,
                playback: {
                    muted: muted,
                    volume: volume,
                },
            }), 'playback is being initialized with expected values');
        });
    });

    QUnit.module('clearQualityRestrictedError', function() {
        QUnit.test('clear QualityRestrictedError action should be correctly formatted', function(assert) {
            const action = PlaybackActions.clearQualityRestrictedError();

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_CLEAR_QUALITY_RESTRICTED_ERROR,
            });
        });
    });

    QUnit.module('qualityIsRestricted', function() {
        QUnit.test('sets QualityRestrictedError action should be correctly formatted', function(assert) {
            const action = PlaybackActions.qualityIsRestricted();

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_QUALITY_RESTRICTED_ERROR,
            });
        });
    });

    QUnit.module('updateDuration', function() {
        QUnit.test('returns an Update Playback State action', function(assert) {
            const duration = parseInt(QUnit.config.current.testId, 36);
            const action = PlaybackActions.updateDuration(duration);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_UPDATE_PLAYBACK_DURATION,
                playback: {
                    duration,
                },
            });
        });
    });

    QUnit.module('setAutoplay', function() {
        QUnit.test('returns a Set Autoplay State action', function(assert) {
            const action = PlaybackActions.setAutoplay(false);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_SET_AUTOPLAY_STATE,
                playback: {
                    autoplay: false,
                },
            });
        });
    });

    QUnit.module('playerMuted', function() {
        QUnit.test('returns a player muted action', function(assert) {
            [true, false].forEach(muted => {
                const action = PlaybackActions.playerMuted(muted);

                assert.deepEqual(action, {
                    type: PlaybackActions.ACTION_PLAYER_MUTED,
                    muted,
                });
            });
        });
    });

    QUnit.module('volumeChanged', function() {
        QUnit.test('returns a volume changed action', function(assert) {
            const volume = Math.random();
            const action = PlaybackActions.volumeChanged(volume);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_VOLUME_CHANGED,
                volume,
            });
        });

        QUnit.test('bounds the volume to a range between 0 and 1', function(assert) {
            const impossiblyQuietVolume = -0.1;
            const impossiblyLoudVolume = 1.1;

            assert.deepEqual(PlaybackActions.volumeChanged(impossiblyQuietVolume), {
                type: PlaybackActions.ACTION_VOLUME_CHANGED,
                volume: 0,
            });

            assert.deepEqual(PlaybackActions.volumeChanged(impossiblyLoudVolume), {
                type: PlaybackActions.ACTION_VOLUME_CHANGED,
                volume: 1,
            });
        });
    });

    QUnit.module('setLoading', function() {
        QUnit.test('returns a setLoading action', function(assert) {
            const action = PlaybackActions.setLoading(true);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_SET_LOADING,
                isLoading: true,
            });
        });
    });

    QUnit.module('updatePlaybackState', function() {
        QUnit.test('with PAUSE returns an ACTION_PAUSE action', function(assert) {
            const action = PlaybackActions.updatePlaybackState(MediaEvents.PAUSE);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_PAUSE,
            });
        });

        QUnit.test('with PLAYING returns an ACTION_PLAYING action', function(assert) {
            const action = PlaybackActions.updatePlaybackState(MediaEvents.PLAYING);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_PLAYING,
            });
        });

        QUnit.test('with WAITING returns an ACTION_WAITING action', function(assert) {
            const action = PlaybackActions.updatePlaybackState(MediaEvents.WAITING);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_WAITING,
            });
        });

        QUnit.test('with ENDED returns an ACTION_ENDED action', function(assert) {
            const action = PlaybackActions.updatePlaybackState(MediaEvents.ENDED);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_ENDED,
            });
        });
    });

    QUnit.module('updateCurrentTime', function() {
        QUnit.test('returns an update current time action', function(assert) {
            const action = PlaybackActions.updateCurrentTime(1);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_UPDATE_CURRENT_TIME,
                currentTime: 1,
            });
        });
    });

    QUnit.module('updateBufferValues', function() {
        QUnit.test('returns an update buffer values action', function(assert) {
            const start = 1;
            const end = 5;
            const action = PlaybackActions.updateBufferValues(start, end);

            assert.deepEqual(action, {
                type: PlaybackActions.ACTION_UPDATE_BUFFER_VALUES,
                start,
                end,
            });
        });
    });

    QUnit.module('incrementBufferEmpties', function() {
        QUnit.test('increments bufferEmpties', function(assert) {
            const action = PlaybackActions.incrementBufferEmpties();

            const expectedAction = {
                type: PlaybackActions.ACTION_INCREMENT_BUFFER_EMPTIES,
            };

            assert.deepEqual(action, expectedAction);
        });
    });

    QUnit.module('setReadyForUserInteraction', function() {
        QUnit.test('returns the ACTION_CONTENT_IS_SHOWING action', function(assert) {
            const action = PlaybackActions.contentIsShowing();
            assert.equal(action.type, PlaybackActions.ACTION_CONTENT_IS_SHOWING);
        });
    });

    QUnit.module('selectRecommendedVideo', function(hooks) {
        hooks.beforeEach(function() {
            this.store = initStore();
            this.fakeWindow = buildFakeWindow({
                location: {
                    search: '?channel=monstercat&otherparam1=blah&otherparam2',
                },
            });
            this.store.dispatch(setWindow(this.fakeWindow));
            this.store.dispatch(setAccessTokenParams({}));
            this.dispatch = sinon.spy();

            this.destinationVod = {
                _id: 'opiuhgvrgthnm',
            };
        });

        QUnit.test('when player is in twitch site, dispatch vodtransition event', function(assert) {
            this.store.dispatch(setPlayerType(PLAYER_SITE));

            const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
            action(this.dispatch, this.store.getState);

            assert.ok(this.dispatch.calledOnce);
            assert.deepEqual(this.dispatch.firstCall.args[0], {
                type: ACTION_TRANSITION_TO_REC_VOD,
                vodId: this.destinationVod._id,
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('when player is in twitch everywhere, dispatch a transition to recommended video, clear existing collection, and setStream with a video', function(assert) {
            this.store.dispatch(setPlayerType(PLAYER_TWITCH_EVERYWHERE));
            const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
            action(this.dispatch, this.store.getState);
            assert.equal(this.dispatch.callCount, 3, 'dispatch was called 3 times');

            const [transitionAction] = this.dispatch.firstCall.args;
            const [clearCollectionThunk] = this.dispatch.secondCall.args;
            const [setStreamThunk] = this.dispatch.thirdCall.args;

            assert.deepEqual(transitionAction, {
                type: ACTION_TRANSITION_TO_REC_VOD,
                vodId: this.destinationVod._id,
            }, 'transition to recommended vod was dispatched');

            this.dispatch.reset();
            clearCollectionThunk(this.dispatch);
            const [clearCollectionAction] = this.dispatch.lastCall.args;
            assert.equal(
                clearCollectionAction.type,
                ACTION_CLEAR_COLLECTION,
                'second dispatch is to clear collection');

            this.dispatch.reset();
            setStreamThunk(this.dispatch, this.store.getState);

            // reset play session, set stream, fetch and set metadata
            assert.equal(this.dispatch.callCount, 6);
            const [setStreamAction] = this.dispatch.thirdCall.args;
            assert.equal(setStreamAction.type, ACTION_SET_STREAM, 'setStream is dispatched');
            assert.equal(setStreamAction.stream.contentType, CONTENT_MODE_VOD, 'setStream to a VOD');
            assert.equal(setStreamAction.stream.videoId, this.destinationVod._id, 'setStream to the correct video');
        });

        QUnit.test('when player is embedded, clear existing collection and setStream with a video', function(assert) {
            this.store.dispatch(setPlayerType(PLAYER_EMBED));

            const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
            action(this.dispatch, this.store.getState);
            assert.equal(this.dispatch.callCount, 2, 'dispatch called twice');

            const [clearCollectionThunk] = this.dispatch.firstCall.args;
            const [setStreamThunk] = this.dispatch.secondCall.args;

            this.dispatch.reset();
            clearCollectionThunk(this.dispatch);
            const [clearCollectionAction] = this.dispatch.lastCall.args;
            assert.equal(
                clearCollectionAction.type,
                ACTION_CLEAR_COLLECTION,
                'first dispatch is to clear collection');

            this.dispatch.reset();
            setStreamThunk(this.dispatch, this.store.getState);

            // reset play session, set stream, fetch and set metadata
            assert.equal(this.dispatch.callCount, 6);
            const [setStreamAction] = this.dispatch.thirdCall.args;
            assert.equal(setStreamAction.type, ACTION_SET_STREAM, 'setStream is dispatched');
            assert.equal(setStreamAction.stream.contentType, CONTENT_MODE_VOD, 'setStream to a VOD');
            assert.equal(setStreamAction.stream.videoId, this.destinationVod._id, 'setStream to the correct video');
        });

        QUnit.module('when player is popout', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch(setPlayerType(PLAYER_POPOUT));
            });

            QUnit.test('change the window url (starting on channel)', function(assert) {
                this.store.dispatch(setWindow(this.fakeWindow));

                const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
                action(this.dispatch, this.store.getState);

                const expectedParams = {
                    video: this.destinationVod._id,
                    otherparam1: 'blah',
                    otherparam2: true,
                };
                const params = parse(this.fakeWindow.location.search.substr(1));
                assert.deepEqual(params, expectedParams, 'video, otherparam, and otherparam1 match');
                assert.equal(this.dispatch.called, false, 'dispatch was not called');
            });

            QUnit.test('change the window url (starting on video)', function(assert) {
                this.fakeWindow = buildFakeWindow({
                    location: {
                        search: '?video=v123456&otherparam1=blah&otherparam2',
                    },
                });
                this.store.dispatch(setWindow(this.fakeWindow));

                const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
                action(this.dispatch, this.store.getState);

                const expectedParams = {
                    video: this.destinationVod._id,
                    otherparam1: 'blah',
                    otherparam2: true,
                };
                const params = parse(this.fakeWindow.location.search.substr(1));
                assert.deepEqual(params, expectedParams, 'video, otherparam1, and otherparam2 match');
                assert.equal(this.dispatch.called, false, 'dispatch was not called');
            });

            QUnit.test('change the window url (starting on collection)', function(assert) {
                this.fakeWindow = buildFakeWindow({
                    location: {
                        search: '?collection=Abc123&otherparam1=blah&otherparam2',
                    },
                });
                this.store.dispatch(setWindow(this.fakeWindow));

                const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
                action(this.dispatch, this.store.getState);

                const expectedParams = {
                    video: this.destinationVod._id,
                    otherparam1: 'blah',
                    otherparam2: true,
                };
                const params = parse(this.fakeWindow.location.search.substr(1));
                assert.deepEqual(params, expectedParams, 'video, otherparam1, and otherparam2 match');
                assert.equal(this.dispatch.called, false, 'dispatch was not called');
            });

            QUnit.test('change the window url (starting on collection and video)', function(assert) {
                this.fakeWindow = buildFakeWindow({
                    location: {
                        search: '?collection=Abc123&video=v123456&otherparam1=blah&otherparam2',
                    },
                });
                this.store.dispatch(setWindow(this.fakeWindow));

                const action = PlaybackActions.selectRecommendedVideo(this.destinationVod._id);
                action(this.dispatch, this.store.getState);

                const expectedParams = {
                    video: this.destinationVod._id,
                    otherparam1: 'blah',
                    otherparam2: true,
                };
                const params = parse(this.fakeWindow.location.search.substr(1));
                assert.deepEqual(params, expectedParams, 'video, otherparam1, and otherparam2 match');
                assert.equal(this.dispatch.called, false, 'dispatch was not called');
            });
        });
    });

    QUnit.module('selectCollectionVideo', function(hooks) {
        hooks.beforeEach(function() {
            this.store = initStore();
            this.fakeWindow = buildFakeWindow({
                location: {
                    search: '?channel=monstercat&otherparam1=blah&otherparam2',
                },
            });
            this.store.dispatch(setWindow(this.fakeWindow));
            this.store.dispatch(setAccessTokenParams({}));
            this.dispatch = sinon.spy();

            this.destinationVod = {
                _id: 'opiuhgvrgthnm',
            };
            this.collection = TEST_COLLECTION;
        });

        QUnit.module('when player is in twitch site', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch(setPlayerType(PLAYER_SITE));
            });

            QUnit.test('when player is fullscreen, dispatch setStream with a video', function(assert) {
                this.store.dispatch(setFullScreen(true));
                const action = PlaybackActions.selectCollectionVideo(this.destinationVod._id, this.collection._id);
                action(this.dispatch, this.store.getState);
                assert.equal(this.dispatch.callCount, 1, 'dispatch called once');

                const [setStreamThunk] = this.dispatch.firstCall.args;
                this.dispatch.reset();
                setStreamThunk(this.dispatch, this.store.getState);

                // reset play session, set stream, fetch and set metadata
                assert.equal(this.dispatch.callCount, 6);
                const [setStreamAction] = this.dispatch.thirdCall.args;
                assert.equal(setStreamAction.type, ACTION_SET_STREAM, 'setStream is dispatched');
                assert.equal(setStreamAction.stream.contentType, CONTENT_MODE_VOD, 'setStream to a VOD');
                assert.equal(setStreamAction.stream.videoId, this.destinationVod._id, 'setStream to the correct video');
            });

            QUnit.test('when player is not fullscreen, dispatch vodtransition event', function(assert) {
                const action = PlaybackActions.selectCollectionVideo(this.destinationVod._id, this.collection._id);
                action(this.dispatch, this.store.getState);

                assert.ok(this.dispatch.calledOnce);
                assert.deepEqual(this.dispatch.firstCall.args[0], {
                    type: ACTION_TRANSITION_TO_COLLECTION,
                    vodId: this.destinationVod._id,
                    collectionId: this.collection._id,
                });
            });
        });

        QUnit.test('when player is embedded, dispatch setStream with a video', function(assert) {
            this.store.dispatch(setPlayerType(PLAYER_EMBED));

            const action = PlaybackActions.selectCollectionVideo(this.destinationVod._id, this.collection._id);
            action(this.dispatch, this.store.getState);
            assert.equal(this.dispatch.callCount, 1, 'dispatch called once');

            const [setStreamThunk] = this.dispatch.firstCall.args;
            this.dispatch.reset();
            setStreamThunk(this.dispatch, this.store.getState);

            // reset play session, set stream, fetch and set metadata
            assert.equal(this.dispatch.callCount, 6);
            const [setStreamAction] = this.dispatch.thirdCall.args;
            assert.equal(setStreamAction.type, ACTION_SET_STREAM, 'setStream is dispatched');
            assert.equal(setStreamAction.stream.contentType, CONTENT_MODE_VOD, 'setStream to a VOD');
            assert.equal(setStreamAction.stream.videoId, this.destinationVod._id, 'setStream to the correct video');
        });

        QUnit.module('when player is popout', function(hooks) {
            hooks.beforeEach(function() {
                this.store.dispatch(setPlayerType(PLAYER_POPOUT));
            });

            QUnit.module('playing a collection', function(hooks) {
                hooks.beforeEach(function() {
                    this.store.dispatch(setCollection(this.collection));
                    this.store.dispatch(loadedCollectionItem());
                });

                QUnit.test('change the window url (collection param only)', function(assert) {
                    this.fakeWindow = buildFakeWindow({
                        location: {
                            search: `?collection=${this.collection._id}&otherparam1=blah&otherparam2`,
                        },
                    });
                    this.store.dispatch(setWindow(this.fakeWindow));

                    const action = PlaybackActions.selectCollectionVideo(
                        this.collection.items[0].item_id,
                        this.collection._id
                    );
                    action(this.dispatch, this.store.getState);

                    const expectedParams = {
                        video: this.collection.items[0].item_id,
                        collection: this.collection._id,
                        otherparam1: 'blah',
                        otherparam2: true,
                    };
                    const params = parse(this.fakeWindow.location.search.substr(1));
                    assert.deepEqual(params, expectedParams, 'collection, video, otherparam1, and otherparam2 match');
                    assert.equal(this.dispatch.called, false, 'dispatch was not called');
                });

                QUnit.test('change the window url (with collection, video, and time params)', function(assert) {
                    this.fakeWindow = buildFakeWindow({
                        location: {
                            // eslint-disable-next-line max-len
                            search: `?collection=${this.collection._id}&video=${this.collection.items[0].item_id}&time=111&otherparam1=blah&otherparam2`,
                        },
                    });
                    this.store.dispatch(setWindow(this.fakeWindow));

                    const action = PlaybackActions.selectCollectionVideo(
                        this.collection.items[1].item_id,
                        this.collection._id
                    );
                    action(this.dispatch, this.store.getState);

                    const expectedParams = {
                        video: this.collection.items[1].item_id,
                        collection: this.collection._id,
                        otherparam1: 'blah',
                        otherparam2: true,
                    };
                    const params = parse(this.fakeWindow.location.search.substr(1));
                    assert.deepEqual(
                        params,
                        expectedParams,
                        'collection, video, otherparam1, and otherparam2 match, no start time provided'
                    );
                    assert.equal(this.dispatch.called, false, 'dispatch was not called');
                });
            });

            QUnit.module('for pending collection request', function(hooks) {
                hooks.beforeEach(function() {
                    this.pendingCollection = TEST_COLLECTION;
                    this.store.dispatch(setPlayerType(PLAYER_POPOUT));
                });

                QUnit.test('change the window url (with a pending collection request)', function(assert) {
                    this.store.dispatch(requestCollection(this.pendingCollection._id));
                    const action = PlaybackActions.selectCollectionVideo(
                        this.pendingCollection.items[0].item_id,
                        this.pendingCollection._id
                    );
                    action(this.dispatch, this.store.getState);

                    const expectedParams = {
                        video: this.pendingCollection.items[0].item_id,
                        collection: this.pendingCollection._id,
                        otherparam1: 'blah',
                        otherparam2: true,
                    };

                    const params = parse(this.fakeWindow.location.search.substr(1));
                    // eslint-disable-next-line max-len
                    assert.deepEqual(params, expectedParams, 'collection, video, otherparam1, and otherparam2 match');
                    assert.equal(this.dispatch.called, false, 'dispatch was not called');
                });

                // eslint-disable-next-line max-len
                QUnit.test('change the window url (with a pending collection request and existing video param)', function(assert) {
                    this.fakeWindow = buildFakeWindow({
                        location: {
                            search: `?video=${this.pendingCollection.items[0].item_id}&otherparam1=blah&otherparam2`,
                        },
                    });
                    this.store.dispatch(setWindow(this.fakeWindow));

                    this.store.dispatch(requestCollection(this.pendingCollection._id));
                    const action = PlaybackActions.selectCollectionVideo(
                        this.pendingCollection.items[0].item_id,
                        this.pendingCollection._id
                    );
                    action(this.dispatch, this.store.getState);

                    const expectedParams = {
                        video: this.pendingCollection.items[0].item_id,
                        collection: this.pendingCollection._id,
                        otherparam1: 'blah',
                        otherparam2: true,
                    };

                    const params = parse(this.fakeWindow.location.search.substr(1));
                    assert.deepEqual(params, expectedParams, 'collection, video, otherparam1, and otherparam2 match');
                    assert.equal(this.dispatch.called, false, 'dispatch was not called');
                });

                // eslint-disable-next-line max-len
                QUnit.test('change the window url (with a pending collection request and existing video, collection, and time params)', function(assert) {
                    // creating separate collection to distinguish from pending collection
                    /* eslint-disable camelcase */
                    const existingCollection = assign({}, TEST_COLLECTION, {
                        _id: QUnit.config.current.testId,
                        items: [
                            {
                                _id: 'fake_vod_id',
                                item_type: 'video',
                                item_id: 'v121212',
                                owner: {
                                    _id: 'fake_user',
                                    display_name: 'FaKeUsEr',
                                    name: 'SomeBigPhony',
                                },
                                title: '',
                                description_html: '',
                                duration: 123123123,
                                views: 121212,
                            },
                        ],
                    });
                    /* eslint-enable camelcase */
                    this.store.dispatch(setCollection(existingCollection));
                    this.fakeWindow = buildFakeWindow({
                        location: {
                            // eslint-disable-next-line max-len
                            search: `?collection=${existingCollection._id}&video=${existingCollection.items[0].item_id}&time=111&otherparam1=blah&otherparam2`,
                        },
                    });
                    this.store.dispatch(setWindow(this.fakeWindow));
                    this.store.dispatch(requestCollection(this.pendingCollection._id));
                    const action = PlaybackActions.selectCollectionVideo(
                        this.pendingCollection.items[0].item_id,
                        this.pendingCollection._id
                    );
                    action(this.dispatch, this.store.getState);

                    const expectedParams = {
                        video: this.pendingCollection.items[0].item_id,
                        collection: this.pendingCollection._id,
                        otherparam1: 'blah',
                        otherparam2: true,
                    };

                    const params = parse(this.fakeWindow.location.search.substr(1));
                    assert.deepEqual(
                        params,
                        expectedParams,
                        'collection, video, otherparam1, and otherparam2 match, no start time provided'
                    );
                    assert.equal(this.dispatch.called, false, 'dispatch was not called');
                });
            });
        });
    });

    QUnit.test('playbackRateChanged returns expected action object', function(assert) {
        const rate = 2;
        const expectedResult = {
            type: PlaybackActions.ACTION_PLAYBACK_RATE_CHANGED,
            playbackRate: rate,
        };

        const actualResult = PlaybackActions.playbackRateChanged(rate);

        assert.deepEqual(actualResult, expectedResult);
    });

    QUnit.test('playerSeeking returns expected action object', function(assert) {
        const expectedResult = {
            type: PlaybackActions.ACTION_PLAYER_SEEKING,
        };

        const actualResult = PlaybackActions.playerSeeking();

        assert.deepEqual(actualResult, expectedResult, 'playerSeeking action is well formed');
    });

    QUnit.test('playerSeeked returns expected action object', function(assert) {
        const time = 99;
        const expectedResult = {
            type: PlaybackActions.ACTION_PLAYER_SEEKED,
            currentTime: time,
        };

        const actualResult = PlaybackActions.playerSeeked(time);

        assert.deepEqual(actualResult, expectedResult, 'playerSeeked action is well formed');
    });

    // eslint-disable-next-line max-len
    QUnit.test('setStartTime returns expected action object when startTime is a number greater than 0', function(assert) {
        const startTime = 100;
        const expectedResult = {
            type: PlaybackActions.ACTION_SET_START_TIME,
            startTime,
        };

        const actualResult = PlaybackActions.setStartTime(startTime);
        assert.deepEqual(actualResult, expectedResult, 'setStarTime action is valid action with start time');
    });

    // eslint-disable-next-line max-len
    QUnit.test('setStartTime returns invalid action object when startTime is not a number or a number < 0', function(assert) {
        const expectedResult = {
            type: PlaybackActions.INVALID_PLAYBACK_ACTION,
        };

        assert.deepEqual(
            PlaybackActions.setStartTime('100'),
            expectedResult,
            'setStarTime action is invalid action on non number start time'
        );
        assert.deepEqual(
            PlaybackActions.setStartTime(-1),
            expectedResult,
            'setStarTime action is invalid action on start time less than 0'
        );
    });
});
