import { unitTest } from 'tests/utils/module';
import sinon from 'sinon';
import find from 'lodash/find';
import assign from 'lodash/assign';
import includes from 'lodash/includes';
import { init as initStore } from 'state';
import { parseUri } from 'util/parseuri';
import { apiHost as API_HOST } from 'settings';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { V3_VIDEOS_RESPONSE,
         NORMALIZED_V3_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import * as RecommendActions from 'actions/recommendations';
import { ACTION_SET_STREAM } from 'actions/stream';
import { setResumeTimes } from 'actions/resume-watch';
import { setVODStreamMetadata } from 'actions/stream-metadata';
import { setWindow } from 'actions/window';
import { setExperiments } from 'actions/experiments';
import { PINEAPPLE } from 'experiments';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { waitFor } from 'tests/utils/waitFor';

unitTest('actions | recommendations', function() {
    QUnit.module('fetchRecommendedVODs', function(hooks) {
        hooks.beforeEach(function() {
            this.state = initStore();
            this.channelName = 'lirik';
            this.setTimeout = sinon.spy();
            const fakeWindow = buildFakeWindow({
                setTimeout: this.setTimeout,
            });
            this.dispatch = sinon.spy();
            this.api.setLoggedIn(true);
            this.api.expectV3VODs(this.channelName, V3_VIDEOS_RESPONSE);

            this.state.dispatch(setWindow(fakeWindow));

            this.createExperiment = ({ pineapple }) => {
                return {
                    get: uuid => {
                        switch (uuid) {
                        case PINEAPPLE:
                            return Promise.resolve(pineapple);
                        }
                    },
                };
            };

            this.setSimilarVODsResponse = videos => {
                const urlregex = /\/kraken\/videos\/similar\/\w+/;

                this.api.mock(urlregex, {
                    status: 200,
                    headers: {},
                    body: {
                        similar_videos: videos, // eslint-disable-line camelcase
                    },
                });
            };

            this.setMockVODStreamMetadata = () => {
                this.state.dispatch(setVODStreamMetadata({
                    // eslint-disable-next-line camelcase
                    broadcast_id: V3_VIDEOS_RESPONSE.videos[0].broadcast_id,
                    channel: {
                        name: this.channelName,
                    },
                }));
            };

            this.setVODStream = vodID => {
                this.state.dispatch({
                    type: ACTION_SET_STREAM,
                    stream: new VODTwitchContentStream(vodID, Promise.resolve({}), {}, {}, {}),
                });
                this.setMockVODStreamMetadata();
            };

            this.getDispatchedRecommendations = () => {
                const dispatchCall = find(this.dispatch.args, arg => {
                    return arg[0].type === RecommendActions.ACTION_SET_RECOMMENDED_VODS;
                });
                return dispatchCall[0].videos;
            };

            this.setRecommendationTypes = (recommendations, type) => {
                return recommendations.map(recommendation => {
                    return assign({}, recommendation, {
                        recommendationType: type,
                    });
                });
            };

            this.state.dispatch(setExperiments(this.createExperiment({
                pineapple: 'no',
            })));
            this.setMockVODStreamMetadata();
        });

        QUnit.test('makes request to channel videos endpoint, if pineapple experiment is not `yes`', function(assert) {
            const expectedArgs = {
                limit: `${RecommendActions.MAX_RECOMMENDED_VODS_VISIBLE + 1}`,
                // eslint-disable-next-line camelcase
                broadcast_type: 'archive',
            };

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                const urlRegex = new RegExp(`${API_HOST}/kraken/channels/${this.channelName}/videos`);
                const request = find(this.api.calls().matched, req => urlRegex.test(req[0]));
                const urlObj = parseUri(request[0]);
                assert.deepEqual(urlObj.queryKey, expectedArgs);
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('makes request to only similar videos endpoint if pineapple experiment is `yes`, and there are similar videos', function(assert) {
            const vodID = 'v123456';

            this.setVODStream(vodID);

            this.setSimilarVODsResponse(V3_VIDEOS_RESPONSE.videos);

            this.state.dispatch(setExperiments(this.createExperiment({
                pineapple: 'yes',
            })));

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                const similarVideoURL = `${API_HOST}/kraken/videos/similar/${vodID}`;
                const requestURLs = this.api.calls().matched.map(req => req[0]);
                assert.ok(includes(requestURLs, similarVideoURL));
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('makes request to channel videos endpoint if pineapple experiment is `yes`, and there are no similar videos', function(assert) {
            const vodID = 'v123456';

            this.setVODStream(vodID);

            this.setSimilarVODsResponse([]);

            this.state.dispatch(setExperiments(this.createExperiment({
                pineapple: 'yes',
            })));

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                const similarVideoURL = `${API_HOST}/kraken/videos/similar/${vodID}`;
                const requestURLs = this.api.calls().matched.map(req => req[0]);
                assert.ok(includes(requestURLs, similarVideoURL));
                return waitFor(() => {
                    let channelVideosURLCalled = false;
                    const urlRegex = new RegExp(`${API_HOST}/kraken/channels/${this.channelName}/videos`);
                    this.api.calls().matched.forEach(req => {
                        channelVideosURLCalled = urlRegex.test(req[0]) || channelVideosURLCalled;
                    });
                    return channelVideosURLCalled;
                });
            });
        });

        QUnit.test('checks fetching vod recommendations set a delay less than MAX_JITTER_DELAY', function(assert) {
            const action = RecommendActions.fetchRecommendedVODs();
            action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                assert.equal(this.setTimeout.callCount, 1);
                assert.equal(this.setTimeout.firstCall.args[1] < RecommendActions.MAX_JITTER_DELAY, true);
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('with post recommendations type passed in, sets recommendations `type` property correctly when fetch recommended videos action successfully finishes', function(assert) {
            const vodID = 'v123456';

            this.setVODStream(vodID);

            const action = RecommendActions.fetchRecommendedVODs(RecommendActions.POST_VOD_RECOMMENDATIONS_TYPE);
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                const dispatchCall = find(this.dispatch.args, arg => {
                    return arg[0].type === RecommendActions.ACTION_SET_RECOMMENDED_VODS;
                });
                assert.equal(dispatchCall[0].recommendationsType, RecommendActions.POST_VOD_RECOMMENDATIONS_TYPE);
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('with offline recommendations type passed in, sets recommendations `type` property correctly when fetch recommended videos action successfully finishes', function(assert) {
            const vodID = 'v123456';

            this.setVODStream(vodID);

            const action = RecommendActions.fetchRecommendedVODs(RecommendActions.OFFLINE_RECOMMENDATIONS_TYPE);
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                const dispatchCall = find(this.dispatch.args, arg => {
                    return arg[0].type === RecommendActions.ACTION_SET_RECOMMENDED_VODS;
                });
                assert.equal(dispatchCall[0].recommendationsType, RecommendActions.OFFLINE_RECOMMENDATIONS_TYPE);
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('sets `recommendationType` property on a recommendation to `similar` when retrieved from similar endpoint', function(assert) {
            const vodID = 'v123456';

            this.setVODStream(vodID);

            this.setSimilarVODsResponse(V3_VIDEOS_RESPONSE.videos);

            this.state.dispatch(setExperiments(this.createExperiment({
                pineapple: 'yes',
            })));

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                this.getDispatchedRecommendations().forEach(rec => {
                    assert.equal(rec.recommendationType, RecommendActions.SIMILAR_VODS);
                });
            });
        });

        // eslint-disable-next-line max-len
        QUnit.test('sets `recommendationType` property on a recommendation to `channel` when retrieved from channel endpoint', function(assert) {
            this.state.dispatch(setExperiments(this.createExperiment({
                pineapple: 'no',
            })));

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                this.getDispatchedRecommendations().forEach(rec => {
                    assert.equal(rec.recommendationType, RecommendActions.CHANNEL_VODS);
                });
            });
        });

        QUnit.test('orders watched and unwatched VOD recommendations correctly', function(assert) {
            this.state.dispatch(setResumeTimes([
                /* eslint-disable camelcase */
                {
                    video_id: V3_VIDEOS_RESPONSE.videos[1]._id,
                    position: V3_VIDEOS_RESPONSE.videos[1].length - 30,
                    type: CONTENT_MODE_VOD,
                }, // watched
                {
                    video_id: V3_VIDEOS_RESPONSE.videos[2]._id,
                    position: V3_VIDEOS_RESPONSE.videos[2].length * 0.05,
                    type: CONTENT_MODE_VOD,
                }, // unwatched
                {
                    video_id: V3_VIDEOS_RESPONSE.videos[3]._id,
                    position: 0,
                    type: CONTENT_MODE_VOD,
                }, // watched
                {
                    video_id: V3_VIDEOS_RESPONSE.videos[4]._id,
                    position: V3_VIDEOS_RESPONSE.videos[4].length * 0.5,
                    type: CONTENT_MODE_VOD,
                }, // watched
                // all other videos are undefined so unwatched
                /* eslint-enable camelcase */
            ]));

            const orderedWatchedRecs = [
                NORMALIZED_V3_RECOMMENDED_VIDEOS[1],
                NORMALIZED_V3_RECOMMENDED_VIDEOS[3],
                NORMALIZED_V3_RECOMMENDED_VIDEOS[4],
            ];

            const orderedUnwatchedRecs = NORMALIZED_V3_RECOMMENDED_VIDEOS.filter(vid => {
                return !includes(orderedWatchedRecs,vid);
            });

            const expectedRecs = this.setRecommendationTypes(
                orderedUnwatchedRecs.concat(orderedWatchedRecs),
                RecommendActions.CHANNEL_VODS
            );

            this.state.dispatch(setVODStreamMetadata({
                // eslint-disable-next-line camelcase
                broadcast_id: '',
                channel: {
                    name: this.channelName,
                },
            }));

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                assert.deepEqual(this.getDispatchedRecommendations(),  expectedRecs);
            });
        });

        QUnit.test('removes VOD matching streamID', function(assert) {
            this.state.dispatch(setVODStreamMetadata({
                // eslint-disable-next-line camelcase
                broadcast_id: V3_VIDEOS_RESPONSE.videos[5].broadcast_id,
                channel: {
                    name: this.channelName,
                },
            }));

            const expectedRecs = this.setRecommendationTypes(
                NORMALIZED_V3_RECOMMENDED_VIDEOS,
                RecommendActions.CHANNEL_VODS
            );

            expectedRecs.splice(5,1);

            const action = RecommendActions.fetchRecommendedVODs();
            const result = action(this.dispatch, this.state.getState);
            return waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
                return result;
            }).then(() => {
                assert.deepEqual(this.getDispatchedRecommendations(),  expectedRecs);
            });
        });

        QUnit.test('sets status to `fetching` before fetching VODs', function(assert) {
            const action = RecommendActions.fetchRecommendedVODs();

            const setFetching = {
                type: RecommendActions.ACTION_SET_FETCHING_STATUS,
                status: RecommendActions.FETCHING,
            };

            const result = action(this.dispatch, this.state.getState);
            waitFor(() => this.setTimeout.called).then(() => {
                this.setTimeout.firstCall.args[0]();
            });
            return result.then(() => {
                assert.ok(this.dispatch.calledTwice);
                assert.ok(this.dispatch.firstCall.calledWith(setFetching));
                assert.ok(this.dispatch.secondCall.calledWith(sinon.match({
                    type: RecommendActions.ACTION_SET_RECOMMENDED_VODS,
                })));
            });
        });

        QUnit.test('will not fetch recommendations if they are already being fetched', function(assert) {
            this.state.dispatch({
                type: RecommendActions.ACTION_SET_FETCHING_STATUS,
                status: RecommendActions.FETCHING,
            });

            const action = RecommendActions.fetchRecommendedVODs();
            return action(this.dispatch, this.state.getState).then(() => {
                assert.ok(!this.dispatch.called);
            });
        });

        QUnit.test('will not fetch recommendations if they have already been fetched', function(assert) {
            this.state.dispatch({
                type: RecommendActions.ACTION_SET_FETCHING_STATUS,
                status: RecommendActions.FETCHED,
            });

            const action = RecommendActions.fetchRecommendedVODs();
            return action(this.dispatch, this.state.getState).then(() => {
                assert.ok(!this.dispatch.called);
            });
        });

        QUnit.test('will not fetch recommendations if no channel name is provided', function(assert) {
            this.state.dispatch(setVODStreamMetadata({
                // eslint-disable-next-line camelcase
                broadcast_id: V3_VIDEOS_RESPONSE.videos[5].broadcast_id,
                channel: {
                    name: '',
                },
            }));

            const action = RecommendActions.fetchRecommendedVODs();
            return action(this.dispatch, this.state.getState).then(() => {
                assert.ok(!this.dispatch.called);
            });
        });
    });

    QUnit.test('setRecommendedVODs is formatted properly', function(assert) {
        const videos = NORMALIZED_V3_RECOMMENDED_VIDEOS.slice(0, 5);
        const recommendationsType = RecommendActions.POST_VOD_RECOMMENDATIONS_TYPE;
        const action = RecommendActions.setRecommendedVODs(videos, recommendationsType);

        assert.deepEqual(action, {
            type: RecommendActions.ACTION_SET_RECOMMENDED_VODS,
            videos,
            recommendationsType,
        });
    });

    QUnit.test('clearRecommendedVODs if formatted properly', function(assert) {
        const action = RecommendActions.clearRecommendedVODs();

        assert.deepEqual(action, {
            type: RecommendActions.ACTION_CLEAR_RECOMMENDED_VODS,
        });
    });

    QUnit.test('setVODVisibility is formatted correctly', function(assert) {
        const visibleVODs = parseInt(QUnit.config.current.testId, 36);
        const expectedResult = {
            type: RecommendActions.ACTION_SET_NUM_VODS_VISIBLE,
            numVids: visibleVODs,
        };
        const actualResult = RecommendActions.setVODVisibility(visibleVODs);

        assert.deepEqual(actualResult, expectedResult);
    });
});
