import sinon from 'sinon';
import { unitTest } from 'tests/utils/module';
import { init as initStore } from 'state';
import { setWindow } from 'actions/window';
import { localStore } from 'tests/fakes/local-store.fake';
import * as ResumeWatchActions from 'actions/resume-watch';
import { UPDATE_INTERVAL_VOD, UPDATE_INTERVAL_LIVESTREAM, INIT_UPDATE_OFFSET_VOD,
         MAX_INIT_UPDATE_OFFSET_LIVESTREAM } from 'state/resume-watch';
import { parse as parseParams } from 'util/params';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { LiveTwitchContentStream, CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { buildFakeWindow } from 'tests/fakes/window.fake';

unitTest('actions/resume-watch', function() {
    QUnit.module('initVodResume', function(hooks) {
        hooks.beforeEach(function() {
            this.videoID = `v${parseInt(QUnit.config.current.testId, 36)}`;
            this.time = Math.floor(Math.random() * 1e3);
            this.broadcastID = `${Math.floor(Math.random() * 1e8)}`;
            this.channelUserID = `${Math.floor(Math.random() * 1e8)}`;
            this.streamTime = Math.floor(Math.random() * 1e3);
            localStore.set('vodResumeTimes', {
                [this.videoID]: this.time,
            });
            localStore.set('vodResumeWatcheds', {
                [this.videoID]: true,
            });
            localStore.set('livestreamResumeTimes', {
                [this.broadcastID]: this.streamTime,
            });
        });
        QUnit.test('should dispatch an Init VOD Resume action', function(assert) {
            const action = ResumeWatchActions.initVodResume();

            assert.deepEqual(action, {
                type: ResumeWatchActions.ACTION_VOD_INIT_RESUME,
                times: {
                    [this.videoID]: this.time,
                },
                watch: {
                    [this.videoID]: true,
                },
                streamTimes: {
                    [this.broadcastID]: this.streamTime,
                },
                lastTimeStamp: 0,
                userId: null,
                isSeeked: false,
            });
        });

        QUnit.test('should dispatch an Init VOD Resume action when a VOD is playing', function(assert) {
            const action = ResumeWatchActions.initVodResume();

            assert.deepEqual(action, {
                type: ResumeWatchActions.ACTION_VOD_INIT_RESUME,
                times: {
                    [this.videoID]: this.time,
                },
                watch: {
                    [this.videoID]: true,
                },
                streamTimes: {
                    [this.broadcastID]: this.streamTime,
                },
                lastTimeStamp: 0,
                userId: null,
                isSeeked: false,
            });
        });

        QUnit.test('should dispatch an Init VOD Resume action when a livestream is playing', function(assert) {
            const action = ResumeWatchActions.initVodResume();

            assert.deepEqual(action, {
                type: ResumeWatchActions.ACTION_VOD_INIT_RESUME,
                times: {
                    [this.videoID]: this.time,
                },
                watch: {
                    [this.videoID]: true,
                },
                streamTimes: {
                    [this.broadcastID]: this.streamTime,
                },
                lastTimeStamp: 0,
                userId: null,
                isSeeked: false,
            });
        });

        QUnit.test('should use sensible values as defaults when the keys don\'t exist', function(assert) {
            localStore.clear();
            const action = ResumeWatchActions.initVodResume();

            assert.deepEqual(action, {
                type: ResumeWatchActions.ACTION_VOD_INIT_RESUME,
                times: {},
                watch: {},
                streamTimes: {},
                lastTimeStamp: 0,
                userId: null,
                isSeeked: false,
            });
        });
    });

    QUnit.test('setResumeTimes dispatches setResumeTimes actions', function(assert) {
        const times = ['time1', 'time2'];
        const action = ResumeWatchActions.setResumeTimes(times);
        assert.deepEqual(action, {
            type: ResumeWatchActions.ACTION_SET_RESUME_TIMES,
            resumeTimes: times,
        });
    });

    QUnit.test('setUser returns the expected action', function(assert) {
        const userId = 123456;
        const action = ResumeWatchActions.setUser(userId);
        assert.deepEqual(action, {
            type: ResumeWatchActions.ACTION_VOD_SET_USER,
            userId,
        });
    });

    QUnit.module('setVodResumeTime', function(hooks) {
        hooks.beforeEach(function() {
            this.videoID = `v${parseInt(QUnit.config.current.testId, 36)}`;
        });

        QUnit.module('if the user is not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.state = {
                    resumeWatch: {
                        userId: null,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };
            });

            QUnit.test('should return a function that dispatches a VOD Set Resume Time action', function(assert) {
                const time = Math.floor(Math.random() * 1e4);
                const action = ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, time);
                const dispatch = sinon.spy();

                action(dispatch, () => this.state);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_VOD_SET_RESUME_TIME,
                    videoID: this.videoID,
                    time,
                }));
            });

            QUnit.test('should return a function that does not set backend time', function(assert) {
                const time = Math.floor(Math.random() * 1e4);
                const action = ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, time);
                const dispatch = sinon.spy();

                action(dispatch, () => this.state);

                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should persist the current vod resume times', function(assert) {
                const time = Math.floor(Math.random() * 1e4);
                const store = initStore();
                const storeWindow = buildFakeWindow({
                    Date: {
                        now() {
                            return 1e8;
                        },

                    },
                });
                store.dispatch(setWindow(storeWindow));
                const action = ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, time);

                store.dispatch(action);

                assert.deepEqual(localStore.get('vodResumeTimes'), {
                    [this.videoID]: time,
                });
                assert.deepEqual(localStore.get('vodResumeWatcheds'), {
                    [this.videoID]: true,
                });
                assert.deepEqual(localStore.get('livestreamResumeTimes'), {});
            });
        });

        QUnit.module('if the user is logged in', function() {
            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a VOD Set Resume Time action when the last time was posted to backend < ${UPDATE_INTERVAL_VOD} seconds ago`, function(assert) {
                const time = 20 + UPDATE_INTERVAL_VOD + 1;
                const action = ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, time);
                const dispatch = sinon.spy();
                const state = {
                    resumeWatch: {
                        userId: '123',
                        lastTimeStamp: 20,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };

                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_VOD_SET_RESUME_TIME,
                    videoID: this.videoID,
                    time,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a VOD Set Resume Time action when time since initTime < ${INIT_UPDATE_OFFSET_VOD}`, function(assert) {
                const time = 20 + UPDATE_INTERVAL_VOD + 1;
                const action = ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, time);
                const dispatch = sinon.spy();
                const state = {
                    resumeWatch: {
                        userId: '123',
                        lastTimeStamp: 20,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD - 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };
                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_VOD_SET_RESUME_TIME,
                    videoID: this.videoID,
                    time,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should return a function that dispatches a VOD Post Backend action only', function(assert) {
                this.api.expectPutResumeWatching('123');
                const time = 20 + UPDATE_INTERVAL_VOD + 1.234;
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, time);
                const state = {
                    resumeWatch: {
                        userId: '123',
                        lastTimeStamp: 20,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };
                action(dispatch, () => state);
                const innerAction = dispatch.firstCall.args[0];
                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_VOD_SET_RESUME_TIME,
                    videoID: this.videoID,
                    time,
                }));
                assert.equal(dispatch.callCount, 1);
                dispatch.reset();

                innerAction(dispatch, () => state);
                assert.ok(dispatch.calledWith({
                    lastTimeStamp: 20 + UPDATE_INTERVAL_VOD + 1,
                    type: ResumeWatchActions.ACTION_VOD_POST_BACKEND_TIME,
                }));
                assert.equal(dispatch.callCount, 1);

                assert.ok(this.api.called('https://api.twitch.tv/api/resumewatching/user-video?id=123'));
                const lastCall = this.api.lastCall('https://api.twitch.tv/api/resumewatching/user-video?id=123');
                assert.equal(lastCall[1].method, 'PUT');
                const requestData = parseParams(lastCall[1].body);
                assert.deepEqual(requestData, {
                    // eslint-disable-next-line camelcase
                    video_id: `${this.videoID}`,
                    position: `${20 + UPDATE_INTERVAL_VOD + 1}`,
                    type: CONTENT_MODE_VOD,
                });
            });
        });
    });

    QUnit.module('cancelVodResumeTime', function(hooks) {
        hooks.beforeEach(function() {
            this.videoID = `v${parseInt(QUnit.config.current.testId, 36)}`;
        });

        QUnit.module('if the user is not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.state = {
                    resumeWatch: {
                        userId: null,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };
            });

            QUnit.test('should return a function that dispatches a VOD Cancel Resume action', function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelVodResumeTime(this.videoID);

                action(dispatch, () => this.state);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_VOD_CANCEL_RESUME,
                    videoID: this.videoID,
                }));
            });

            QUnit.test('should return a function that does not post backend time', function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelVodResumeTime(this.videoID);

                action(dispatch, () => this.state);

                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should persist the current vod resume watching state', function(assert) {
                const cancelTime = Math.floor(Math.random() * 1e3);
                const okVideoID = `v${Math.floor(Math.random() * 1e6)}`;
                const okVideoTime = Math.floor(Math.random() * 1e3);
                const store = initStore();
                const storeWindow = buildFakeWindow({
                    Date: {
                        now() {
                            return 1e8;
                        },

                    },
                });
                store.dispatch(setWindow(storeWindow));

                store.dispatch(ResumeWatchActions.setVodResumeTime(this.videoID, this.channelUserID, cancelTime));
                store.dispatch(ResumeWatchActions.setVodResumeTime(okVideoID, this.channelUserID, okVideoTime));

                assert.deepEqual(localStore.get('vodResumeTimes'), {
                    [this.videoID]: cancelTime,
                    [okVideoID]: okVideoTime,
                });
                assert.deepEqual(localStore.get('vodResumeWatcheds'), {
                    [this.videoID]: true,
                    [okVideoID]: true,
                });
                assert.deepEqual(localStore.get('livestreamResumeTimes'), {});
                const action = ResumeWatchActions.cancelVodResumeTime(this.videoID);

                store.dispatch(action);

                assert.deepEqual(localStore.get('vodResumeTimes'), {
                    [okVideoID]: okVideoTime,
                });
                assert.deepEqual(localStore.get('vodResumeWatcheds'), {
                    [okVideoID]: true,
                });
                assert.deepEqual(localStore.get('livestreamResumeTimes'), {});
            });
        });

        QUnit.module('if the user is logged in', function() {
            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a VOD Cancel Resume Time action when the last time was posted to backend < ${UPDATE_INTERVAL_VOD}`, function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelVodResumeTime(this.videoID);
                const state = {
                    resumeWatch: {
                        userId: '1234',
                        lastTimeStamp: 19,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };
                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_VOD_CANCEL_RESUME,
                    videoID: this.videoID,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a VOD Cancel Resume action when time since initTime < ${INIT_UPDATE_OFFSET_VOD} seconds ago`, function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelVodResumeTime(this.videoID);

                action(dispatch, () => ({
                    resumeWatch: {
                        userId: '1234',
                        lastTimeStamp: 21,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD - 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                }));

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_VOD_CANCEL_RESUME,
                    videoID: this.videoID,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should return a function that dispatches a VOD post backend action only', function(assert) {
                this.api.expectPutResumeWatching('1234');
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelVodResumeTime(this.videoID);
                const state = {
                    resumeWatch: {
                        userId: '1234',
                        lastTimeStamp: 21,
                        initUpdateOffset: INIT_UPDATE_OFFSET_VOD,
                        updateInterval: UPDATE_INTERVAL_VOD,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + INIT_UPDATE_OFFSET_VOD + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new VODTwitchContentStream(),
                };
                action(dispatch, () => state);
                const innerAction = dispatch.firstCall.args[0];
                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_VOD_CANCEL_RESUME,
                    videoID: this.videoID,
                }));
                assert.equal(dispatch.callCount, 1);
                dispatch.reset();

                innerAction(dispatch, () => state);
                assert.ok(dispatch.calledWith({
                    lastTimeStamp: 0,
                    type: ResumeWatchActions.ACTION_VOD_POST_BACKEND_TIME,
                }));
                assert.ok(this.api.called('https://api.twitch.tv/api/resumewatching/user-video?id=1234'));
                const lastCall = this.api.lastCall('https://api.twitch.tv/api/resumewatching/user-video?id=1234');
                assert.equal(lastCall[1].method, 'PUT');
                const requestData = parseParams(lastCall[1].body);
                assert.deepEqual(requestData, {
                    // eslint-disable-next-line camelcase
                    video_id: `${this.videoID}`,
                    position: '0',
                    type: CONTENT_MODE_VOD,
                });
                assert.equal(dispatch.callCount, 1);
            });
        });
    });

    QUnit.module('setLivestreamResumeTime', function(hooks) {
        hooks.beforeEach(function() {
            this.broadcastID = `${parseInt(QUnit.config.current.testId, 36)}`;
        });

        QUnit.module('if the user is not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.state = {
                    resumeWatch: {
                        userId: null,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + MAX_INIT_UPDATE_OFFSET_LIVESTREAM + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };
            });

            // eslint-disable-next-line max-len
            QUnit.test('should return a function that dispatches a Livestream Set Resume Time action', function(assert) {
                const time = Math.floor(Math.random() * 1e4);
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelUserID, time);

                action(dispatch, () => this.state);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_SET_RESUME_TIME,
                    broadcastID: this.broadcastID,
                    time,
                }));
            });

            QUnit.test('should return a function that does not post backend time', function(assert) {
                const time = Math.floor(Math.random() * 1e4);
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelUserID, time);

                action(dispatch, () => this.state);

                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should persist the current livestream resume times', function(assert) {
                const time = Math.floor(Math.random() * 1e4);
                const store = initStore();
                const action = ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelUserID, time);
                const storeWindow = buildFakeWindow({
                    Date: {
                        now() {
                            return 1e8;
                        },

                    },
                });
                store.dispatch(setWindow(storeWindow));

                store.dispatch(action);

                assert.deepEqual(localStore.get('vodResumeTimes'), {});
                assert.deepEqual(localStore.get('vodResumeWatcheds'), {});
                assert.deepEqual(localStore.get('livestreamResumeTimes'), {
                    [this.broadcastID]: time,
                });
            });
        });

        QUnit.module('if the user is logged in', function() {
            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a Livestream Set Resume Time action when the last time was posted to backend < ${UPDATE_INTERVAL_LIVESTREAM} seconds ago`, function(assert) {
                const time = 300 + UPDATE_INTERVAL_VOD - 1;
                const action = ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelUserID, time);
                const dispatch = sinon.spy();
                const initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
                const state = {
                    resumeWatch: {
                        userId: '123',
                        lastTimeStamp: 300,
                        initUpdateOffset,
                        updateInterval: UPDATE_INTERVAL_LIVESTREAM,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + initUpdateOffset + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };

                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_SET_RESUME_TIME,
                    broadcastID: this.broadcastID,
                    time,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that only dispatches a Livestream Set Resume Time action when time since initTime < ${MAX_INIT_UPDATE_OFFSET_LIVESTREAM}`, function(assert) {
                const time = 300 + UPDATE_INTERVAL_LIVESTREAM + 1;
                const action = ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelUserID, time);
                const dispatch = sinon.spy();
                const initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
                const state = {
                    resumeWatch: {
                        userId: '123',
                        lastTimeStamp: 300,
                        initUpdateOffset,
                        updateInterval: UPDATE_INTERVAL_LIVESTREAM,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + initUpdateOffset - 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };

                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_SET_RESUME_TIME,
                    broadcastID: this.broadcastID,
                    time,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            // eslint-disable-next-line max-len
            QUnit.test('should return a function that dispatches a Livestream Post Backend action only', function(assert) {
                this.api.expectPutResumeWatching('123');
                const time = 300 + UPDATE_INTERVAL_LIVESTREAM + 1.234;
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.setLivestreamResumeTime(this.broadcastID, this.channelUserID, time);
                const initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
                const state = {
                    resumeWatch: {
                        userId: '123',
                        lastTimeStamp: 300,
                        initUpdateOffset,
                        updateInterval: UPDATE_INTERVAL_LIVESTREAM,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + initUpdateOffset + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };

                action(dispatch, () => state);
                const innerAction = dispatch.firstCall.args[0];
                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_SET_RESUME_TIME,
                    broadcastID: this.broadcastID,
                    time,
                }));
                assert.equal(dispatch.callCount, 1);
                dispatch.reset();

                innerAction(dispatch, () => state);
                assert.ok(dispatch.calledWith({
                    lastTimeStamp: 300 + UPDATE_INTERVAL_LIVESTREAM + 1,
                    type: ResumeWatchActions.ACTION_VOD_POST_BACKEND_TIME,
                }));
                assert.ok(this.api.called('https://api.twitch.tv/api/resumewatching/user-video?id=123'));
                const lastCall = this.api.lastCall('https://api.twitch.tv/api/resumewatching/user-video?id=123');
                assert.equal(lastCall[1].method, 'PUT');
                const requestData = parseParams(lastCall[1].body);
                assert.deepEqual(requestData, {
                    // eslint-disable-next-line camelcase
                    video_id: `${this.broadcastID}`,
                    position: `${300 + UPDATE_INTERVAL_LIVESTREAM + 1}`,
                    type: CONTENT_MODE_LIVE,
                });
                assert.equal(dispatch.callCount, 1);
            });
        });
    });

    QUnit.module('cancelLivestreamResumeTime', function(hooks) {
        hooks.beforeEach(function() {
            this.broadcastID = `${parseInt(QUnit.config.current.testId, 36)}`;
        });

        QUnit.module('if the user is not logged in', function(hooks) {
            hooks.beforeEach(function() {
                this.state = {
                    resumeWatch: {
                        userId: null,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + MAX_INIT_UPDATE_OFFSET_LIVESTREAM + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };
            });

            QUnit.test('should return a function that dispatches a Livestream Cancel Resume action', function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID);

                action(dispatch, () => this.state);

                assert.ok(dispatch.calledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
            });

            QUnit.test('should return a function that does not post backend time', function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID);

                action(dispatch, () => this.state);

                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should persist the current livestream resume watching state', function(assert) {
                const cancelTime = Math.floor(Math.random() * 1e3);
                const okBroadcastID = `${Math.floor(Math.random() * 1e6)}`;
                const okBroadcastTime = Math.floor(Math.random() * 1e3);
                const store = initStore();
                const storeWindow = buildFakeWindow({
                    Date: {
                        now() {
                            return 1e8;
                        },

                    },
                });
                store.dispatch(setWindow(storeWindow));

                store.dispatch(ResumeWatchActions.setLivestreamResumeTime(
                    this.broadcastID,
                    this.channelUserID,
                    cancelTime
                ));
                store.dispatch(ResumeWatchActions.setLivestreamResumeTime(
                    okBroadcastID,
                    this.channelUserID,
                    okBroadcastTime
                ));

                assert.deepEqual(localStore.get('vodResumeTimes'), {});
                assert.deepEqual(localStore.get('vodResumeWatcheds'), {});
                assert.deepEqual(localStore.get('livestreamResumeTimes'), {
                    [this.broadcastID]: cancelTime,
                    [okBroadcastID]: okBroadcastTime,
                });

                const action = ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID);

                store.dispatch(action);

                assert.deepEqual(localStore.get('vodResumeTimes'), {});
                assert.deepEqual(localStore.get('vodResumeWatcheds'), {});
                assert.deepEqual(localStore.get('livestreamResumeTimes'), {
                    [okBroadcastID]: okBroadcastTime,
                });
            });
        });

        QUnit.module('if the user is logged in', function() {
            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a VOD Set Resume Time action when the last time was posted to backend < ${UPDATE_INTERVAL_LIVESTREAM} seconds ago`, function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID);
                const initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
                const state = {
                    resumeWatch: {
                        userId: '1234',
                        lastTimeStamp: UPDATE_INTERVAL_LIVESTREAM - 1,
                        initUpdateOffset,
                        updateInterval: UPDATE_INTERVAL_LIVESTREAM,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + initUpdateOffset + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };

                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            // eslint-disable-next-line max-len
            QUnit.test(`should not return a function that dispatches a Livestream Cancel Resume action when time since initTime < ${MAX_INIT_UPDATE_OFFSET_LIVESTREAM}`, function(assert) {
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID);
                const initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
                const state = {
                    resumeWatch: {
                        userId: '1234',
                        lastTimeStamp: 301,
                        initUpdateOffset,
                        updateInterval: UPDATE_INTERVAL_LIVESTREAM,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + initUpdateOffset - 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };

                action(dispatch, () => state);

                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
                assert.equal(dispatch.callCount, 1);
                assert.equal(this.api.calls().unmatched.length, 0, 'backend time not set');
            });

            QUnit.test('should return a function that dispatches a VOD Post Backend Action only', function(assert) {
                this.api.expectPutResumeWatching('1234');
                const dispatch = sinon.spy();
                const action = ResumeWatchActions.cancelLivestreamResumeTime(this.broadcastID);
                const initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
                const state = {
                    resumeWatch: {
                        userId: '1234',
                        lastTimeStamp: 601,
                        initUpdateOffset,
                        updateInterval: UPDATE_INTERVAL_LIVESTREAM,
                    },
                    window: {
                        Date: {
                            now() {
                                return 1e8 + initUpdateOffset + 1;
                            },
                        },
                    },
                    analytics: {
                        playSessionStartTime: 1e8,
                    },
                    stream: new LiveTwitchContentStream(),
                };

                action(dispatch, () => state);
                const innerAction = dispatch.firstCall.args[0];
                assert.ok(dispatch.neverCalledWith({
                    type: ResumeWatchActions.ACTION_LIVESTREAM_CANCEL_RESUME,
                    broadcastID: this.broadcastID,
                }));
                assert.equal(dispatch.callCount, 1);
                dispatch.reset();

                innerAction(dispatch, () => state);
                assert.ok(dispatch.calledWith({
                    lastTimeStamp: 0,
                    type: ResumeWatchActions.ACTION_VOD_POST_BACKEND_TIME,
                }));
                assert.ok(this.api.called('https://api.twitch.tv/api/resumewatching/user-video?id=1234'));
                const lastCall = this.api.lastCall('https://api.twitch.tv/api/resumewatching/user-video?id=1234');
                assert.equal(lastCall[1].method, 'PUT');
                const requestData = parseParams(lastCall[1].body);
                assert.deepEqual(requestData, {
                    // eslint-disable-next-line camelcase
                    video_id: `${this.broadcastID}`,
                    position: '0',
                    type: CONTENT_MODE_LIVE,
                });
                assert.equal(dispatch.callCount, 1);
            });
        });
    });
});
